import { Op, DataTypes } from 'sequelize';
import { SYNC_DIRECTIONS } from '@tamanu/constants';
import { safeJsonParse } from '@tamanu/utils/safeJsonParse';
import { log } from '@tamanu/shared/services/logging';
import { Model } from './Model';
export class SurveyScreenComponent extends Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            screenIndex: DataTypes.INTEGER,
            componentIndex: DataTypes.INTEGER,
            text: DataTypes.STRING,
            visibilityCriteria: DataTypes.STRING,
            validationCriteria: DataTypes.TEXT,
            detail: DataTypes.STRING,
            config: DataTypes.STRING,
            options: DataTypes.TEXT,
            calculation: DataTypes.STRING,
            visibilityStatus: DataTypes.STRING
        }, {
            ...options,
            syncDirection: SYNC_DIRECTIONS.PULL_FROM_CENTRAL
        });
    }
    static getListReferenceAssociations(includeAllVitals) {
        return {
            model: this.sequelize.models.ProgramDataElement,
            as: 'dataElement',
            paranoid: !includeAllVitals
        };
    }
    static initRelations(models) {
        this.belongsTo(models.Survey, {
            foreignKey: 'surveyId'
        });
        this.belongsTo(models.ProgramDataElement, {
            foreignKey: 'dataElementId',
            as: 'dataElement'
        });
    }
    static async getComponentsForSurveys(surveyIds, options = {}) {
        const { includeAllVitals } = options;
        const where = {
            surveyId: {
                [Op.in]: surveyIds
            }
        };
        const components = await this.findAll({
            where,
            include: this.getListReferenceAssociations(includeAllVitals),
            order: [
                [
                    'screen_index',
                    'ASC'
                ],
                [
                    'component_index',
                    'ASC'
                ]
            ],
            paranoid: !includeAllVitals
        });
        return components.map((c)=>c.forResponse());
    }
    static getComponentsForSurvey(surveyId, options = {}) {
        return this.getComponentsForSurveys([
            surveyId
        ], options);
    }
    getOptions() {
        try {
            const optionString = this.options || this.dataElement?.defaultOptions || '';
            if (!optionString) {
                return [];
            }
            const optionArray = JSON.parse(optionString);
            return Object.entries(optionArray).map(([label, value])=>({
                    label,
                    value
                }));
        } catch (e) {
            log.error(e);
            return [];
        }
    }
    forResponse() {
        const { options, ...values } = this.dataValues;
        return {
            ...values,
            options: safeJsonParse(options)
        };
    }
    static buildSyncFilter() {
        return null; // syncs everywhere
    }
    static buildSyncLookupQueryDetails() {
        return null; // syncs everywhere
    }
}

//# sourceMappingURL=SurveyScreenComponent.js.map