"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "notes", {
    enumerable: true,
    get: function() {
        return noteRoute;
    }
});
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _errors = require("@tamanu/shared/errors");
const _constants = require("@tamanu/constants");
const _checkNotePermission = require("../../../utils/checkNotePermission");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const noteRoute = _express.default.Router();
// Encounter notes cannot be edited
function canModifyNote(note) {
    return note.recordType !== _constants.NOTE_RECORD_TYPES.ENCOUNTER;
}
noteRoute.post('/$', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, body: noteData } = req;
    await (0, _checkNotePermission.checkNotePermission)(req, noteData, 'create');
    const note = await models.Note.create({
        recordType: noteData.recordType,
        recordId: noteData.recordId,
        date: noteData.date,
        noteType: noteData.noteType,
        authorId: noteData.authorId,
        onBehalfOfId: noteData.onBehalfOfId,
        revisedById: noteData.revisedById,
        content: noteData.content.trim()
    });
    res.send(note);
}));
noteRoute.get('/:id', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params } = req;
    const noteId = params.id;
    const note = await models.Note.findOne({
        include: [
            {
                model: models.User,
                as: 'author'
            },
            {
                model: models.User,
                as: 'onBehalfOf'
            }
        ],
        where: {
            id: noteId,
            visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT
        }
    });
    await (0, _checkNotePermission.checkNotePermission)(req, note, 'read');
    res.send(note);
}));
noteRoute.put('/:id', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, body, params } = req;
    const editedNote = await models.Note.findOne({
        include: [
            {
                model: models.User,
                as: 'author'
            },
            {
                model: models.User,
                as: 'onBehalfOf'
            }
        ],
        where: {
            id: params.id,
            visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT
        }
    });
    if (!editedNote) {
        throw new _errors.NotFoundError();
    }
    if (canModifyNote(editedNote) === false) {
        throw new _errors.ForbiddenError('Cannot edit encounter notes.');
    }
    req.checkPermission('write', editedNote.recordType);
    const owner = await models[editedNote.recordType].findByPk(editedNote.recordId);
    req.checkPermission('write', owner);
    await editedNote.update({
        ...body
    });
    res.send(editedNote);
}));
noteRoute.delete('/:id', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models } = req;
    const noteToDelete = await models.Note.findOne({
        where: {
            id: req.params.id,
            visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT
        }
    });
    if (!noteToDelete) {
        throw new _errors.NotFoundError();
    }
    if (canModifyNote(noteToDelete) === false) {
        throw new _errors.ForbiddenError('Cannot delete encounter notes.');
    }
    req.checkPermission('write', noteToDelete.recordType);
    await noteToDelete.update({
        visibilityStatus: _constants.VISIBILITY_STATUSES.HISTORICAL
    });
    res.send({});
}));

//# sourceMappingURL=notes.js.map