"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "suggestions", {
    enumerable: true,
    get: function() {
        return suggestions;
    }
});
const _case = require("case");
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _sequelize = require("sequelize");
const _errors = require("@tamanu/shared/errors");
const _lodash = require("lodash");
const _constants = require("@tamanu/constants");
const _uuid = require("uuid");
const _nanoid = require("nanoid");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const suggestions = _express.default.Router();
const defaultLimit = 25;
const defaultMapper = ({ name, code, id })=>({
        name,
        code,
        id
    });
// Translation helpers
const extractDataId = ({ stringId })=>stringId.split('.').pop();
const replaceDataLabelsWithTranslations = ({ data, translations })=>{
    const translationsByDataId = (0, _lodash.keyBy)(translations, extractDataId);
    return data.map((item)=>{
        const itemData = item instanceof _sequelize.Sequelize.Model ? item.dataValues : item; // if is Sequelize model, use the dataValues instead to prevent Converting circular structure to JSON error when destructing
        return {
            ...itemData,
            name: translationsByDataId[item.id]?.text || item.name
        };
    });
};
const ENDPOINT_TO_DATA_TYPE = {
    // Special cases where the endpoint name doesn't match the dataType
    ['facilityLocationGroup']: _constants.OTHER_REFERENCE_TYPES.LOCATION_GROUP,
    ['bookableLocationGroup']: _constants.OTHER_REFERENCE_TYPES.LOCATION_GROUP,
    ['patientLabTestCategories']: _constants.REFERENCE_TYPES.LAB_TEST_CATEGORY,
    ['patientLabTestPanelTypes']: _constants.OTHER_REFERENCE_TYPES.LAB_TEST_PANEL,
    ['invoiceProducts']: _constants.OTHER_REFERENCE_TYPES.INVOICE_PRODUCT
};
const getDataType = (endpoint)=>ENDPOINT_TO_DATA_TYPE[endpoint] || endpoint;
function createSuggesterRoute(endpoint, modelName, whereBuilder, { mapper, searchColumn, extraReplacementsBuilder, includeBuilder, orderBuilder }) {
    suggestions.get(`/${endpoint}$`, (0, _expressasynchandler.default)(async (req, res)=>{
        req.checkPermission('list', modelName);
        const { models, query } = req;
        const { language } = query;
        delete query.language;
        const model = models[modelName];
        const searchQuery = (query.q || '').trim().toLowerCase();
        const positionQuery = (0, _sequelize.literal)(`POSITION(LOWER(:positionMatch) in LOWER(${`"${modelName}"."${searchColumn}"`})) > 1`);
        const dataType = getDataType(endpoint);
        const isTranslatable = _constants.TRANSLATABLE_REFERENCE_TYPES.includes(dataType);
        const translations = isTranslatable ? await models.TranslatedString.getReferenceDataTranslationsByDataType({
            language,
            refDataType: dataType,
            queryString: searchQuery,
            limit: defaultLimit
        }) : [];
        const suggestedIds = translations.map(extractDataId);
        const whereQuery = whereBuilder(`%${searchQuery}%`, query, req);
        const where = {
            [_sequelize.Op.or]: [
                whereQuery,
                {
                    // Wrap inside AND block to avoid being overwritten by whereQuery results
                    [_sequelize.Op.and]: {
                        id: {
                            [_sequelize.Op.in]: suggestedIds
                        }
                    },
                    ...(0, _lodash.omit)(whereQuery, 'name')
                }
            ]
        };
        if (endpoint === 'location' && query.locationGroupId) {
            where.locationGroupId = query.locationGroupId;
        }
        const include = includeBuilder?.(req);
        const order = orderBuilder?.(req);
        const results = await model.findAll({
            where,
            include,
            order: [
                ...order ? [
                    order
                ] : [],
                positionQuery,
                [
                    _sequelize.Sequelize.literal(`"${modelName}"."${searchColumn}"`),
                    'ASC'
                ]
            ],
            replacements: {
                positionMatch: searchQuery,
                ...extraReplacementsBuilder(query)
            },
            limit: defaultLimit
        });
        // Allow for async mapping functions (currently only used by location suggester)
        const data = await Promise.all(results.map((r)=>mapper(r)));
        res.send(isTranslatable ? replaceDataLabelsWithTranslations({
            data,
            translations
        }) : data);
    }));
}
// this exists so a control can look up the associated information of a given suggester endpoint
// when it's already been given an id so that it's guaranteed to have the same structure as the
// options endpoint
function createSuggesterLookupRoute(endpoint, modelName, { mapper }) {
    suggestions.get(`/${endpoint}/:id`, (0, _expressasynchandler.default)(async (req, res)=>{
        const { models, params, query: { language = _constants.ENGLISH_LANGUAGE_CODE } } = req;
        req.checkPermission('list', modelName);
        const record = await models[modelName].findByPk(params.id);
        if (!record) throw new _errors.NotFoundError();
        req.checkPermission('read', record);
        const mappedRecord = await mapper(record);
        if (!_constants.TRANSLATABLE_REFERENCE_TYPES.includes(getDataType(endpoint))) {
            res.send(mappedRecord);
            return;
        }
        const translation = await models.TranslatedString.findOne({
            where: {
                stringId: `${_constants.REFERENCE_DATA_TRANSLATION_PREFIX}.${getDataType(endpoint)}.${record.id}`,
                language
            },
            attributes: [
                'stringId',
                'text'
            ],
            raw: true
        });
        if (!translation) {
            res.send(mappedRecord);
            return;
        }
        const translatedRecord = replaceDataLabelsWithTranslations({
            data: [
                mappedRecord
            ],
            translations: [
                translation
            ]
        })[0];
        res.send(translatedRecord);
    }));
}
function createAllRecordsRoute(endpoint, modelName, whereBuilder, { mapper, searchColumn, extraReplacementsBuilder }) {
    suggestions.get(`/${endpoint}/all$`, (0, _expressasynchandler.default)(async (req, res)=>{
        req.checkPermission('list', modelName);
        const { models, query } = req;
        const model = models[modelName];
        const where = whereBuilder('%', query, req);
        const results = await model.findAll({
            where,
            order: [
                [
                    _sequelize.Sequelize.literal(searchColumn),
                    'ASC'
                ]
            ],
            replacements: extraReplacementsBuilder(query)
        });
        const mappedResults = await Promise.all(results.map(mapper));
        if (!_constants.TRANSLATABLE_REFERENCE_TYPES.includes(getDataType(endpoint))) {
            res.send(mappedResults);
            return;
        }
        const translatedStrings = await models.TranslatedString.getReferenceDataTranslationsByDataType({
            language: query.language,
            refDataType: getDataType(endpoint)
        });
        const translatedResults = replaceDataLabelsWithTranslations({
            data: mappedResults,
            translations: translatedStrings
        });
        // Allow for async mapping functions (currently only used by location suggester)
        res.send(translatedResults);
    }));
}
function createSuggesterCreateRoute(endpoint, modelName, { creatingBodyBuilder, mapper, afterCreated }) {
    suggestions.post(`/${endpoint}/create`, (0, _expressasynchandler.default)(async (req, res)=>{
        const { models } = req;
        req.checkPermission('create', modelName);
        const body = await creatingBodyBuilder(req);
        const newRecord = await models[modelName].create(body, {
            returning: true
        });
        if (afterCreated) {
            await afterCreated(req, newRecord);
        }
        const mappedRecord = await mapper(newRecord);
        res.send(mappedRecord);
    }));
}
// Add a new suggester for a particular model at the given endpoint.
// Records will be filtered based on the whereSql parameter. The user's search term
// will be passed to the sql query as ":search" - see the existing suggestion
// endpoints for usage examples.
function createSuggester(endpoint, modelName, whereBuilder, optionOverrides, allowCreatingNewSuggestion) {
    const options = {
        mapper: defaultMapper,
        searchColumn: 'name',
        extraReplacementsBuilder: ()=>{},
        ...optionOverrides
    };
    // Note: createAllRecordsRoute and createSuggesterLookupRoute must
    // be added in this order otherwise the :id param will match all
    createAllRecordsRoute(endpoint, modelName, whereBuilder, options);
    createSuggesterLookupRoute(endpoint, modelName, options);
    createSuggesterRoute(endpoint, modelName, whereBuilder, options);
    if (allowCreatingNewSuggestion) {
        createSuggesterCreateRoute(endpoint, modelName, options);
    }
}
// this should probably be changed to a `visibility_criteria IN ('list', 'of', 'statuses')`
// once there's more than one status that we're checking against
const VISIBILITY_CRITERIA = {
    visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT
};
const afterCreatedReferenceData = async (req, newRecord)=>{
    const { models } = req;
    if (newRecord.type === _constants.REFERENCE_TYPES.TASK_TEMPLATE) {
        await models.TaskTemplate.create({
            referenceDataId: newRecord.id
        });
    }
};
const referenceDataBodyBuilder = ({ type, name })=>{
    if (!name) {
        throw new _errors.ValidationError('Name is required');
    }
    if (!type) {
        throw new _errors.ValidationError('Type is required');
    }
    const code = `${(0, _lodash.camelCase)(name)}-${(0, _nanoid.customAlphabet)('1234567890ABCDEFGHIJKLMNPQRSTUVWXYZ', 3)()}`;
    return {
        id: (0, _uuid.v4)(),
        code,
        type,
        name
    };
};
createSuggester('multiReferenceData', 'ReferenceData', (search, { types })=>({
        type: {
            [_sequelize.Op.in]: types
        },
        name: {
            [_sequelize.Op.iLike]: search
        },
        ...VISIBILITY_CRITERIA
    }), {
    includeBuilder: (req)=>{
        const { models: { ReferenceData, TaskTemplate }, query: { relationType } } = req;
        if (!relationType) return undefined;
        return [
            {
                model: TaskTemplate,
                as: 'taskTemplate',
                include: TaskTemplate.getFullReferenceAssociations()
            },
            {
                model: ReferenceData,
                as: 'children',
                required: false,
                through: {
                    attributes: [],
                    where: {
                        type: relationType,
                        deleted_at: null
                    }
                },
                include: {
                    model: TaskTemplate,
                    as: 'taskTemplate',
                    include: TaskTemplate.getFullReferenceAssociations()
                },
                where: VISIBILITY_CRITERIA
            }
        ];
    },
    orderBuilder: (req)=>{
        const { query } = req;
        const types = query.types;
        if (!types?.length) return;
        const caseStatement = query.types.map((value, index)=>`WHEN '${value}' THEN ${index + 1}`).join(' ');
        return [
            _sequelize.Sequelize.literal(`
        CASE "ReferenceData"."type"
          ${caseStatement}
          ELSE ${query.types.length + 1}
        END
      `)
        ];
    },
    mapper: (item)=>item,
    creatingBodyBuilder: (req)=>referenceDataBodyBuilder({
            type: req.body.type,
            name: req.body.name
        }),
    afterCreated: afterCreatedReferenceData
}, true);
_constants.REFERENCE_TYPE_VALUES.forEach((typeName)=>{
    createSuggester(typeName, 'ReferenceData', (search)=>({
            name: {
                [_sequelize.Op.iLike]: search
            },
            type: typeName,
            ...VISIBILITY_CRITERIA
        }), {
        includeBuilder: (req)=>{
            const { models: { ReferenceData }, query: { parentId, relationType = _constants.DEFAULT_HIERARCHY_TYPE } } = req;
            if (!parentId) return undefined;
            return {
                model: ReferenceData,
                as: 'parent',
                required: true,
                through: {
                    attributes: [],
                    where: {
                        referenceDataParentId: parentId,
                        type: relationType
                    }
                }
            };
        },
        creatingBodyBuilder: (req)=>referenceDataBodyBuilder({
                type: typeName,
                name: req.body.name
            }),
        afterCreated: afterCreatedReferenceData
    }, true);
});
createSuggester('labTestType', 'LabTestType', ()=>VISIBILITY_CRITERIA, {
    mapper: ({ name, code, id, labTestCategoryId })=>({
            name,
            code,
            id,
            labTestCategoryId
        })
});
const DEFAULT_WHERE_BUILDER = (search)=>({
        name: {
            [_sequelize.Op.iLike]: search
        },
        ...VISIBILITY_CRITERIA
    });
const filterByFacilityWhereBuilder = (search, query)=>{
    const baseWhere = DEFAULT_WHERE_BUILDER(search);
    // Parameters are passed as strings, so we need to check for 'true'
    const shouldFilterByFacility = query.filterByFacility === 'true' || query.filterByFacility === true;
    if (!shouldFilterByFacility) {
        return baseWhere;
    }
    return {
        ...baseWhere,
        facilityId: query.facilityId
    };
};
const createNameSuggester = (endpoint, modelName = (0, _case.pascal)(endpoint), whereBuilderFn = DEFAULT_WHERE_BUILDER, options)=>createSuggester(endpoint, modelName, whereBuilderFn, {
        mapper: ({ id, name })=>({
                id,
                name
            }),
        ...options
    });
createNameSuggester('department', 'Department', filterByFacilityWhereBuilder);
createNameSuggester('facility');
// Calculate the availability of the location before passing on to the front end
createSuggester('location', 'Location', // Allow filtering by parent location group
(search, query)=>{
    const baseWhere = filterByFacilityWhereBuilder(search, query);
    const { ...filters } = query;
    delete filters.q;
    delete filters.filterByFacility;
    if (!query.parentId) {
        return {
            ...baseWhere,
            ...filters
        };
    }
    return {
        ...baseWhere,
        parentId: query.parentId
    };
}, {
    mapper: async (location)=>{
        const availability = await location.getAvailability();
        const { name, code, id, maxOccupancy, facilityId } = location;
        const lg = await location.getLocationGroup();
        const locationGroup = lg && {
            name: lg.name,
            code: lg.code,
            id: lg.id
        };
        return {
            name,
            code,
            maxOccupancy,
            id,
            availability,
            facilityId,
            ...locationGroup && {
                locationGroup
            }
        };
    }
});
createNameSuggester('locationGroup', 'LocationGroup', filterByFacilityWhereBuilder);
// Location groups filtered by facility. Used in the survey form autocomplete
createNameSuggester('facilityLocationGroup', 'LocationGroup', (search, query)=>filterByFacilityWhereBuilder(search, {
        ...query,
        filterByFacility: true
    }));
// Location groups filtered by isBookable. Used in location bookings view
createNameSuggester('bookableLocationGroup', 'LocationGroup', (search, query)=>({
        ...filterByFacilityWhereBuilder(search, {
            ...query,
            filterByFacility: true
        }),
        isBookable: true
    }));
createNameSuggester('survey', 'Survey', (search, { programId })=>({
        name: {
            [_sequelize.Op.iLike]: search
        },
        ...programId ? {
            programId
        } : programId,
        surveyType: {
            [_sequelize.Op.notIn]: [
                _constants.SURVEY_TYPES.OBSOLETE,
                _constants.SURVEY_TYPES.VITALS
            ]
        }
    }));
createSuggester('invoiceProducts', 'InvoiceProduct', (search)=>({
        name: {
            [_sequelize.Op.iLike]: search
        },
        '$referenceData.type$': _constants.REFERENCE_TYPES.ADDITIONAL_INVOICE_PRODUCT,
        ...VISIBILITY_CRITERIA
    }), {
    mapper: (product)=>{
        product.addVirtualFields();
        return product;
    },
    includeBuilder: (req)=>{
        return [
            {
                model: req.models.ReferenceData,
                as: 'referenceData',
                attributes: [
                    'code',
                    'type'
                ]
            }
        ];
    }
});
createSuggester('practitioner', 'User', (search)=>({
        displayName: {
            [_sequelize.Op.iLike]: search
        },
        ...VISIBILITY_CRITERIA
    }), {
    mapper: ({ id, displayName })=>({
            id,
            name: displayName
        }),
    searchColumn: 'display_name'
});
// Remove whitespace from the start and end of each string then combine with a space in between
// E.g. 'William ' + 'Horoto' => 'William Horoto'
const trimAndConcat = (col1, col2)=>_sequelize.Sequelize.fn('concat', _sequelize.Sequelize.fn('trim', _sequelize.Sequelize.col(col1)), ' ', _sequelize.Sequelize.fn('trim', _sequelize.Sequelize.col(col2)));
createSuggester('patient', 'Patient', (search)=>({
        [_sequelize.Op.or]: [
            _sequelize.Sequelize.where(trimAndConcat('first_name', 'last_name'), {
                [_sequelize.Op.iLike]: search
            }),
            _sequelize.Sequelize.where(trimAndConcat('last_name', 'first_name'), {
                [_sequelize.Op.iLike]: search
            }),
            {
                displayId: {
                    [_sequelize.Op.iLike]: search
                }
            }
        ]
    }), {
    mapper: (patient)=>patient,
    searchColumn: 'first_name',
    orderBuilder: (req)=>{
        const searchQuery = (req.query.q || '').trim().toLowerCase();
        const escapedQuery = req.db.escape(searchQuery);
        const escapedPartialMatch = req.db.escape(`${searchQuery}%`);
        return _sequelize.Sequelize.literal(`
          CASE
            WHEN LOWER(display_id) = ${escapedQuery} THEN 0
            WHEN LOWER(display_id) LIKE ${escapedPartialMatch} THEN 1
            WHEN LOWER(TRIM(first_name) || ' ' || TRIM(last_name)) LIKE ${escapedPartialMatch} THEN 2
            WHEN LOWER(TRIM(last_name) || ' ' || TRIM(first_name)) LIKE ${escapedPartialMatch} THEN 3
            ELSE 4
          END
        `);
    }
});
// Specifically fetches lab test categories that have a lab request against a patient
createSuggester('patientLabTestCategories', 'ReferenceData', (search, query, req)=>{
    const baseWhere = DEFAULT_WHERE_BUILDER(search);
    if (!query.patientId) {
        return {
            ...baseWhere,
            type: _constants.REFERENCE_TYPES.LAB_TEST_CATEGORY
        };
    }
    const idBaseFilter = {
        [_sequelize.Op.in]: _sequelize.Sequelize.literal(`(
          SELECT DISTINCT(lab_test_category_id)
          FROM lab_requests
          INNER JOIN
            encounters ON encounters.id = lab_requests.encounter_id
          WHERE lab_requests.status = :lab_request_status
            AND encounters.patient_id = :patient_id
            AND encounters.deleted_at is null
            AND lab_requests.deleted_at is null
        )`)
    };
    const canListSensitive = req.ability.can('list', 'SensitiveLabRequest');
    const idSensitiveFilter = {
        [_sequelize.Op.in]: _sequelize.Sequelize.literal(`(
          SELECT DISTINCT(lab_test_types.lab_test_category_id)
          FROM lab_requests
          INNER JOIN encounters
            ON (encounters.id = lab_requests.encounter_id)
          INNER JOIN lab_tests
            ON (lab_requests.id = lab_tests.lab_request_id)
          INNER JOIN lab_test_types
            ON (lab_test_types.id = lab_tests.lab_test_type_id)
          WHERE lab_requests.status = :lab_request_status
            AND encounters.patient_id = :patient_id
            AND encounters.deleted_at is null
            AND lab_requests.deleted_at is null
            AND lab_test_types.is_sensitive IS FALSE
        )`)
    };
    return {
        ...baseWhere,
        type: _constants.REFERENCE_TYPES.LAB_TEST_CATEGORY,
        id: canListSensitive ? idBaseFilter : idSensitiveFilter
    };
}, {
    extraReplacementsBuilder: (query)=>({
            lab_request_status: query?.status || 'published',
            patient_id: query.patientId
        })
});
// Specifically fetches lab panels that have a lab test against a patient
createSuggester('patientLabTestPanelTypes', 'LabTestPanel', (search, query)=>{
    const baseWhere = DEFAULT_WHERE_BUILDER(search);
    if (!query.patientId) {
        return baseWhere;
    }
    return {
        ...baseWhere,
        id: {
            [_sequelize.Op.in]: _sequelize.Sequelize.literal(`(
          SELECT DISTINCT(lab_test_panel_id)
          FROM lab_test_panel_lab_test_types
          INNER JOIN
            lab_test_types ON lab_test_types.id = lab_test_panel_lab_test_types.lab_test_type_id
          INNER JOIN
            lab_tests ON lab_tests.lab_test_type_id = lab_test_types.id
          INNER JOIN
            lab_requests ON lab_requests.id = lab_tests.lab_request_id
          INNER JOIN
            encounters ON encounters.id = lab_requests.encounter_id
          WHERE lab_requests.status = :lab_request_status
            AND encounters.patient_id = :patient_id
            AND encounters.deleted_at is null
            AND lab_requests.deleted_at is null
        )`)
        }
    };
}, {
    extraReplacementsBuilder: (query)=>({
            lab_request_status: query?.status || 'published',
            patient_id: query.patientId
        })
});
createNameSuggester('programRegistryClinicalStatus', 'ProgramRegistryClinicalStatus', (search, { programRegistryId })=>({
        ...DEFAULT_WHERE_BUILDER(search),
        ...programRegistryId ? {
            programRegistryId
        } : {}
    }));
createSuggester('programRegistry', 'ProgramRegistry', (search, query)=>{
    const baseWhere = DEFAULT_WHERE_BUILDER(search);
    if (!query.patientId) {
        return baseWhere;
    }
    return {
        ...baseWhere,
        // Only suggest program registries this patient isn't already part of
        id: {
            [_sequelize.Op.notIn]: _sequelize.Sequelize.literal(`(
          SELECT DISTINCT(pr.id)
          FROM program_registries pr
          INNER JOIN patient_program_registrations ppr
          ON ppr.program_registry_id = pr.id
          WHERE
            ppr.patient_id = :patient_id
          AND
            ppr.registration_status != '${_constants.REGISTRATION_STATUSES.RECORDED_IN_ERROR}'
        )`)
        }
    };
}, {
    extraReplacementsBuilder: (query)=>({
            patient_id: query.patientId
        })
});
createNameSuggester('programRegistryClinicalStatus', 'ProgramRegistryClinicalStatus', (search, { programRegistryId })=>({
        ...DEFAULT_WHERE_BUILDER(search),
        ...programRegistryId ? {
            programRegistryId
        } : {}
    }));
createNameSuggester('programRegistryCondition', 'ProgramRegistryCondition', (search, { programRegistryId })=>({
        ...DEFAULT_WHERE_BUILDER(search),
        ...programRegistryId ? {
            programRegistryId
        } : {}
    }));
createNameSuggester('programRegistry', 'ProgramRegistry', (search, query)=>{
    const baseWhere = DEFAULT_WHERE_BUILDER(search);
    if (!query.patientId) {
        return baseWhere;
    }
    return {
        ...baseWhere,
        // Only suggest program registries this patient isn't already part of
        id: {
            [_sequelize.Op.notIn]: _sequelize.Sequelize.literal(`(
          SELECT DISTINCT(pr.id)
          FROM program_registries pr
          INNER JOIN patient_program_registrations ppr
          ON ppr.program_registry_id = pr.id
          WHERE
            ppr.patient_id = '${query.patientId}'
          AND
            ppr.registration_status = '${_constants.REGISTRATION_STATUSES.ACTIVE}'
        )`)
        }
    };
});
// TODO: Use generic LabTest permissions for this suggester
createNameSuggester('labTestPanel', 'LabTestPanel');
createNameSuggester('template', 'Template', (search, query)=>{
    const baseWhere = DEFAULT_WHERE_BUILDER(search);
    const { type } = query;
    if (!type) {
        return baseWhere;
    }
    return {
        ...baseWhere,
        type
    };
});
const routerEndpoints = suggestions.stack.map((layer)=>{
    const path = layer.route.path.replace('/', '').replaceAll('$', '');
    const root = path.split('/')[0];
    return root;
});
const rootElements = [
    ...new Set(routerEndpoints)
];
_constants.SUGGESTER_ENDPOINTS.forEach((endpoint)=>{
    if (!rootElements.includes(endpoint)) {
        throw new Error(`Suggester endpoint exists in shared constant but not included in router: ${endpoint}`);
    }
});
rootElements.forEach((endpoint)=>{
    if (!_constants.SUGGESTER_ENDPOINTS.includes(endpoint)) {
        throw new Error(`Suggester endpoint not added to shared constant: ${endpoint}`);
    }
});

//# sourceMappingURL=suggestions.js.map