"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    constructPermission: function() {
        return constructPermission;
    },
    ensurePermissionCheck: function() {
        return ensurePermissionCheck;
    },
    getPermissions: function() {
        return getPermissions;
    },
    getSubjectName: function() {
        return getSubjectName;
    }
});
const _errors = require("../errors");
const _rolesToPermissions = require("./rolesToPermissions");
function getSubjectName(subject) {
    if (!subject || typeof subject === 'string') {
        return subject;
    }
    const Type = typeof subject === 'object' ? subject.constructor : subject;
    return Type.modelName || Type.name;
}
async function constructPermission(req, res, next) {
    try {
        // eslint-disable-next-line require-atomic-updates
        req.ability = await (0, _rolesToPermissions.getAbilityForUser)(req.models, req.user);
        next();
    } catch (e) {
        next(e);
    }
}
const checkIfHasPermission = (req, action, subject, field = '')=>{
    if (req.flagPermissionChecked) {
        req.flagPermissionChecked();
    }
    // allow a null permission to let things through - this means all endpoints
    // still need an explicit permission check, even if it's a null one!
    if (!action) {
        return;
    }
    const { ability } = req;
    if (!ability) {
        // user must log in - 401
        throw new _errors.BadAuthenticationError('No permission');
    }
    if (req.audit) {
        req.audit.addPermissionCheck(action, getSubjectName(subject), subject?.id);
    }
    return ability.can(action, subject, field);
};
function ensurePermissionCheck(req, res, next) {
    const originalResSend = res.send;
    req.checkPermission = (action, subject)=>{
        const hasPermission = checkIfHasPermission(req, action, subject);
        if (!hasPermission) {
            const rule = req.ability.relevantRuleFor(action, subject);
            const reason = rule && rule.reason || `No permission to perform action "${action}" on "${getSubjectName(subject)}"`;
            if (req.audit) {
                req.audit.annotate({
                    forbiddenReason: reason
                });
            }
            throw new _errors.ForbiddenError(reason);
        }
    };
    req.checkForOneOfPermissions = (actions, subject)=>{
        const permissionChecks = actions.map((action)=>checkIfHasPermission(req, action, subject));
        const hasPermission = permissionChecks.some(Boolean);
        if (!hasPermission) {
            const reason = `No permission to perform any of actions "${actions.join(', ')}" on "${getSubjectName(subject)}"`;
            if (req.audit) {
                req.audit.annotate({
                    forbiddenReason: reason
                });
            }
            throw new _errors.ForbiddenError(reason);
        }
    };
    req.checkListOrReadPermission = (subject)=>{
        req.checkForOneOfPermissions([
            'list',
            'read'
        ], subject);
    };
    req.flagPermissionChecked = ()=>{
        res.send = originalResSend;
    };
    res.send = ()=>{
        res.send = originalResSend;
        res.status(501).send({
            error: {
                name: 'NoPermissionCheckError',
                message: 'No permission check was implemented for this endpoint.'
            }
        });
    };
    next();
}
async function getPermissions(req, res, _next) {
    const { user, models } = req;
    const permissions = await (0, _rolesToPermissions.getPermissionsForRoles)(models, user.role);
    res.send({
        permissions
    });
}

//# sourceMappingURL=middleware.js.map