"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "startAllCommand", {
    enumerable: true,
    get: function() {
        return startAllCommand;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _commander = require("commander");
const _logging = require("@tamanu/shared/services/logging");
const _timeZoneCheck = require("@tamanu/shared/utils/timeZoneCheck");
const _selectFacilityIds = require("@tamanu/utils/selectFacilityIds");
const _checkConfig = require("../checkConfig");
const _initDeviceId = require("../sync/initDeviceId");
const _database = require("../database");
const _sync = require("../sync");
const _createApiApp = require("../createApiApp");
const _tasks = require("../tasks");
const _serverInfo = require("../serverInfo");
const _ApplicationContext = require("../ApplicationContext");
const _createSyncApp = require("../createSyncApp");
const _SyncTask = require("../tasks/SyncTask");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
async function startAll({ skipMigrationCheck }) {
    _logging.log.info(`Starting facility server version ${_serverInfo.version}`, {
        serverFacilityIds: (0, _selectFacilityIds.selectFacilityIds)(_config.default)
    });
    _logging.log.info(`Process info`, {
        execArgs: process.execArgs || '<empty>'
    });
    const context = await new _ApplicationContext.ApplicationContext().init({
        appType: 'api'
    });
    if (_config.default.db.migrateOnStartup) {
        await context.sequelize.migrate('up');
    } else {
        await context.sequelize.assertUpToDate({
            skipMigrationCheck
        });
    }
    await (0, _initDeviceId.initDeviceId)(context);
    await (0, _checkConfig.checkConfig)(context);
    await (0, _database.performDatabaseIntegrityChecks)(context);
    context.centralServer = new _sync.CentralServerConnection(context);
    context.syncManager = new _sync.FacilitySyncManager(context);
    context.syncConnection = new _sync.FacilitySyncConnection();
    await (0, _timeZoneCheck.performTimeZoneChecks)({
        remote: context.centralServer,
        sequelize: context.sequelize,
        config: _config.default
    });
    const { server } = await (0, _createApiApp.createApiApp)(context);
    let { port } = _config.default;
    if (+process.env.PORT) {
        port = +process.env.PORT;
    }
    server.listen(port, ()=>{
        _logging.log.info(`API Server is running on port ${port}!`);
    });
    const { server: syncServer } = await (0, _createSyncApp.createSyncApp)(context);
    let { port: syncPort } = _config.default.sync.syncApiConnection;
    syncServer.listen(syncPort, ()=>{
        _logging.log.info(`SYNC server is running on port ${syncPort}!`);
    });
    const syncTaskClass = [
        _SyncTask.SyncTask
    ];
    const cancelTasks = (0, _tasks.startScheduledTasks)(context);
    const cancelSyncTask = (0, _tasks.startScheduledTasks)(context, syncTaskClass);
    process.once('SIGTERM', ()=>{
        _logging.log.info('Received SIGTERM, closing HTTP server');
        cancelTasks();
        cancelSyncTask();
        server.close();
        syncServer.close();
    });
}
const startAllCommand = new _commander.Command('startAll').description('Start both the Tamanu Facility API server, sync server, and tasks runner').option('--skipMigrationCheck', 'skip the migration check on startup').action(startAll);

//# sourceMappingURL=startAll.js.map