"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "login", {
    enumerable: true,
    get: function() {
        return login;
    }
});
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _bcrypt = /*#__PURE__*/ _interop_require_default(require("bcrypt"));
const _jsonwebtoken = /*#__PURE__*/ _interop_require_default(require("jsonwebtoken"));
const _constants = require("@tamanu/constants");
const _auth = require("@tamanu/constants/auth");
const _errors = require("@tamanu/shared/errors");
const _rolesToPermissions = require("@tamanu/shared/permissions/rolesToPermissions");
const _localisation = require("../localisation");
const _convertDbRecord = require("../convertDbRecord");
const _utils = require("./utils");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const getRefreshToken = async (models, { refreshSecret, userId, deviceId })=>{
    const { RefreshToken } = models;
    const { auth, canonicalHostName } = _config.default;
    const { saltRounds, refreshToken: { refreshIdLength, tokenDuration: refreshTokenDuration } } = auth;
    const refreshId = await (0, _utils.getRandomBase64String)(refreshIdLength);
    const refreshTokenJwtId = (0, _utils.getRandomU32)();
    const [hashedRefreshId, refreshToken] = await Promise.all([
        _bcrypt.default.hash(refreshId, saltRounds),
        (0, _utils.buildToken)({
            userId,
            refreshId
        }, refreshSecret, {
            expiresIn: refreshTokenDuration,
            audience: _auth.JWT_TOKEN_TYPES.REFRESH,
            issuer: canonicalHostName,
            jwtid: `${refreshTokenJwtId}`
        })
    ]);
    // Extract expiry as set by jwt.sign
    const { exp } = _jsonwebtoken.default.decode(refreshToken);
    await RefreshToken.upsert({
        refreshId: hashedRefreshId,
        expiresAt: new Date(exp * 1000),
        userId,
        deviceId
    }, {
        where: {
            userId,
            deviceId
        }
    });
    return refreshToken;
};
const login = ({ secret, refreshSecret })=>(0, _expressasynchandler.default)(async (req, res)=>{
        const { store, body, settings } = req;
        const { models } = store;
        const { email, password, facilityIds, deviceId } = body;
        const tamanuClient = req.header('X-Tamanu-Client');
        const getSettingsForFrontEnd = async ()=>{
            // Only attach central scoped settings if login request is for central admin panel login
            if ([
                _constants.SERVER_TYPES.WEBAPP,
                _constants.SERVER_TYPES.MOBILE
            ].includes(tamanuClient) && !facilityIds) {
                return await settings.getFrontEndSettings();
            }
        };
        if (!email || !password) {
            throw new _errors.BadAuthenticationError('Missing credentials');
        }
        const internalClient = (0, _utils.isInternalClient)(tamanuClient);
        if (internalClient && !deviceId) {
            throw new _errors.BadAuthenticationError('Missing deviceId');
        }
        const user = await models.User.getForAuthByEmail(email);
        if (!user && _config.default.auth.reportNoUserError) {
            // an attacker can use this to get a list of user accounts
            // but hiding this error entirely can make debugging a hassle
            // so we just put it behind a config flag
            throw new _errors.BadAuthenticationError('No such user');
        }
        const hashedPassword = user?.password || '';
        if (!await _bcrypt.default.compare(password, hashedPassword)) {
            throw new _errors.BadAuthenticationError('Invalid credentials');
        }
        const { auth, canonicalHostName } = _config.default;
        const { tokenDuration } = auth;
        const accessTokenJwtId = (0, _utils.getRandomU32)();
        const [token, refreshToken, allowedFacilities, localisation, permissions, role] = await Promise.all([
            (0, _utils.buildToken)({
                userId: user.id,
                deviceId
            }, secret, {
                expiresIn: tokenDuration,
                audience: _auth.JWT_TOKEN_TYPES.ACCESS,
                issuer: canonicalHostName,
                jwtid: `${accessTokenJwtId}`
            }),
            internalClient ? getRefreshToken(models, {
                refreshSecret,
                userId: user.id,
                deviceId
            }) : undefined,
            user.allowedFacilities(),
            (0, _localisation.getLocalisation)(),
            (0, _rolesToPermissions.getPermissionsForRoles)(models, user.role),
            models.Role.findByPk(user.role)
        ]);
        // Send some additional data with login to tell the user about
        // the context they've just logged in to.
        res.send({
            token,
            refreshToken,
            user: (0, _convertDbRecord.convertFromDbRecord)((0, _utils.stripUser)(user.get({
                plain: true
            }))).data,
            permissions,
            serverType: _constants.SERVER_TYPES.CENTRAL,
            role: role?.forResponse() ?? null,
            allowedFacilities,
            localisation,
            centralHost: _config.default.canonicalHostName,
            settings: await getSettingsForFrontEnd()
        });
    });

//# sourceMappingURL=login.js.map