"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "LocalSystemFact", {
    enumerable: true,
    get: function() {
        return LocalSystemFact;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _Model = require("./Model");
const _nodecrypto = require("node:crypto");
let LocalSystemFact = class LocalSystemFact extends _Model.Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            // use a separate key to allow for future changes in allowable id format
            key: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false
            },
            value: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: true
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.DO_NOT_SYNC,
            indexes: [
                {
                    unique: true,
                    fields: [
                        'key'
                    ]
                }
            ]
        });
    }
    static async get(key) {
        const result = await this.findOne({
            where: {
                key
            }
        });
        return result?.value;
    }
    static async set(key, value) {
        const existing = await this.findOne({
            where: {
                key
            }
        });
        if (existing) {
            await this.update({
                value
            }, {
                where: {
                    key
                }
            });
        } else {
            // This function is used in the migration code, and in Postgres
            // version 12 `gen_random_uuid()` is not available in a blank
            // database, and it's used to default the ID. So instead, create
            // random UUIDs here in code, so the default isn't invoked. We
            // use Node's native function so it's just as fast.
            await this.create({
                id: (0, _nodecrypto.randomUUID)(),
                key,
                value
            });
        }
    }
    static async incrementValue(key, amount = 1) {
        const [rowsAffected] = await this.sequelize.query(`
        UPDATE
          local_system_facts
        SET
          value = value::integer + :amount,
          updated_at = NOW()
        WHERE
          key = :key
        RETURNING
          value;
      `, {
            replacements: {
                key,
                amount
            }
        });
        if (rowsAffected.length === 0) {
            throw new Error(`The local system fact table does not include the fact ${key}`);
        }
        const fact = rowsAffected[0];
        return fact.value;
    }
};

//# sourceMappingURL=LocalSystemFact.js.map