"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "getValues", {
    enumerable: true,
    get: function() {
        return getValues;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _datefns = require("date-fns");
const _constants = require("@tamanu/constants");
const _fhirTypes = require("@tamanu/shared/services/fhirTypes");
const _fhir = require("@tamanu/shared/utils/fhir");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
async function getValues(upstream, models) {
    const { Encounter } = models;
    if (upstream instanceof Encounter) return getValuesFromEncounter(upstream, models);
    throw new Error(`Invalid upstream type for encounter ${upstream.constructor.name}`);
}
async function getValuesFromEncounter(upstream, models) {
    const subject = await subjectRef(upstream, models);
    const serviceProvider = await serviceProviderRef(upstream, models);
    return {
        lastUpdated: new Date(),
        status: status(upstream),
        class: classification(upstream),
        actualPeriod: period(upstream),
        subject,
        location: await locationRef(upstream, models),
        serviceProvider,
        resolved: subject.isResolved() && (serviceProvider ? serviceProvider.isResolved() : true)
    };
}
function status(encounter) {
    if (encounter.discharge) {
        return _constants.FHIR_ENCOUNTER_STATUS.DISCHARGED;
    }
    return _constants.FHIR_ENCOUNTER_STATUS.IN_PROGRESS;
}
function classification(encounter) {
    const code = classificationCode(encounter);
    if (!code) return [];
    return [
        new _fhirTypes.FhirCodeableConcept({
            coding: [
                new _fhirTypes.FhirCoding({
                    system: _config.default.hl7.dataDictionaries.encounterClass,
                    code,
                    display: _constants.FHIR_ENCOUNTER_CLASS_DISPLAY[code] ?? null
                })
            ]
        })
    ];
}
function classificationCode({ encounterType }) {
    switch(encounterType){
        case _constants.ENCOUNTER_TYPES.ADMISSION:
        case _constants.ENCOUNTER_TYPES.CLINIC:
        case _constants.ENCOUNTER_TYPES.IMAGING:
            return _constants.FHIR_ENCOUNTER_CLASS_CODE.IMP;
        case _constants.ENCOUNTER_TYPES.EMERGENCY:
        case _constants.ENCOUNTER_TYPES.TRIAGE:
            return _constants.FHIR_ENCOUNTER_CLASS_CODE.EMER;
        case _constants.ENCOUNTER_TYPES.OBSERVATION:
            return _constants.FHIR_ENCOUNTER_CLASS_CODE.OBSENC;
        case _constants.ENCOUNTER_TYPES.SURVEY_RESPONSE:
        case _constants.ENCOUNTER_TYPES.VACCINATION:
        default:
            return null; // these should be filtered out (TODO EPI-452)
    }
}
function period(encounter) {
    const start = (0, _datefns.parseISO)(encounter.startDate);
    let end = encounter.endDate;
    if (end) {
        end = (0, _datefns.parseISO)(end);
        if (end <= start) {
            // should never happen in real usage, but test, imported, or migrated data
            // may do this; in that case satisfy Period's requirement that end > start.
            end = (0, _datefns.addSeconds)(start, 1);
        }
    }
    return new _fhirTypes.FhirPeriod({
        start: (0, _fhir.formatFhirDate)(start),
        end: end ? (0, _fhir.formatFhirDate)(end) : null
    });
}
function subjectRef(encounter, models) {
    return _fhirTypes.FhirReference.to(models.FhirPatient, encounter.patient?.id, {
        display: `${encounter.patient?.firstName} ${encounter.patient?.lastName}`
    });
}
const { BED, WARD, JURISDICTION } = _constants.FHIR_LOCATION_PHYSICAL_TYPE_CODE;
async function locationRef(encounter, models) {
    const department = await models.Department.findOne({
        where: {
            id: encounter.departmentId
        }
    });
    return [
        new _fhirTypes.FhirEncounterLocation({
            location: new _fhirTypes.FhirReference({
                display: department?.name,
                id: department?.id
            }),
            status: _constants.FHIR_ENCOUNTER_LOCATION_STATUS.ACTIVE,
            physicalType: new _fhirTypes.FhirCodeableConcept({
                coding: [
                    {
                        system: _config.default.hl7.dataDictionaries.locationPhysicalType,
                        code: JURISDICTION,
                        display: _constants.FHIR_LOCATION_PHYSICAL_TYPE_DISPLAY[JURISDICTION]
                    }
                ]
            })
        }),
        new _fhirTypes.FhirEncounterLocation({
            location: new _fhirTypes.FhirReference({
                display: encounter.location?.locationGroup?.name,
                id: encounter.location?.locationGroup?.id
            }),
            status: _constants.FHIR_ENCOUNTER_LOCATION_STATUS.ACTIVE,
            physicalType: new _fhirTypes.FhirCodeableConcept({
                coding: [
                    {
                        system: _config.default.hl7.dataDictionaries.locationPhysicalType,
                        code: WARD,
                        display: _constants.FHIR_LOCATION_PHYSICAL_TYPE_DISPLAY[WARD]
                    }
                ]
            })
        }),
        new _fhirTypes.FhirEncounterLocation({
            location: new _fhirTypes.FhirReference({
                display: encounter.location?.name,
                id: encounter.location?.id
            }),
            status: _constants.FHIR_ENCOUNTER_LOCATION_STATUS.ACTIVE,
            physicalType: new _fhirTypes.FhirCodeableConcept({
                coding: [
                    {
                        system: _config.default.hl7.dataDictionaries.locationPhysicalType,
                        code: BED,
                        display: _constants.FHIR_LOCATION_PHYSICAL_TYPE_DISPLAY[BED]
                    }
                ]
            })
        })
    ];
}
async function serviceProviderRef(encounter, models) {
    const { facility } = encounter.location || {};
    if (!facility) {
        return null;
    }
    return _fhirTypes.FhirReference.to(models.FhirOrganization, facility.id, {
        display: facility.name
    });
}

//# sourceMappingURL=getValues.js.map