"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.parsePackageRef = parsePackageRef;
exports.readPackage = readPackage;
exports.writePackage = writePackage;
exports.findPackage = findPackage;
exports.findPackageDir = findPackageDir;
exports.findPackageDependencyDir = findPackageDependencyDir;
exports.resolveDependencyChain = resolveDependencyChain;
const fs = __importStar(require("fs"));
const path = __importStar(require("path"));
const path_1 = require("./path");
/**
 * Parse a package reference string. An example reference is the `name`
 * property found in `package.json`.
 *
 * @param r Package reference string
 * @returns Parsed package reference object
 */
function parsePackageRef(r) {
    if (r.startsWith("@")) {
        // If `/` is not found, the reference could be a path alias.
        const indexSeparator = r.indexOf("/");
        if (indexSeparator >= 0) {
            // The separator must have at least 1 character following it, before the
            // end of the string. Note that the scope may be an empty string.
            // TypeScript does not place any restrictions on import re-mappings.
            if (indexSeparator + 1 >= r.length) {
                throw new Error(`Invalid package reference: ${r}`);
            }
            return {
                scope: r.substring(0, indexSeparator),
                name: r.substring(indexSeparator + 1),
            };
        }
    }
    if (!r) {
        throw new Error(`Invalid package reference: "${r}"`);
    }
    return { name: r };
}
/**
 * Resolve a package path to a file reference by appending `package.json`, if needed.
 *
 * @param pkgPath Package path. May contain `package.json`.
 * @returns File reference to `package.json`
 */
function resolvePackagePath(pkgPath) {
    if (path.basename(pkgPath).toLowerCase() !== "package.json") {
        return path.join(pkgPath, "package.json");
    }
    return pkgPath;
}
/**
 * Read a `package.json` manifest from a file.
 *
 * @param pkgPath Either a path directly to the target `package.json` file, or the directory containing it.
 * @returns Package manifest
 */
function readPackage(pkgPath) {
    const pkgFile = resolvePackagePath(pkgPath);
    return JSON.parse(fs.readFileSync(pkgFile, "utf-8"));
}
/**
 * Write a `package.json` manifest to a file.
 *
 * @param pkgPath Either a path directly to the target `package.json` file, or the directory containing it.
 * @param manifest Package manifest
 * @param space Indentation to apply to the output
 */
function writePackage(pkgPath, manifest, space = "  ") {
    const pkgFile = resolvePackagePath(pkgPath);
    fs.writeFileSync(pkgFile, JSON.stringify(manifest, undefined, space) + "\n", "utf-8");
}
/**
 * Find the nearest `package.json` manifest file. Search upward through all
 * parent directories.
 *
 * If a starting directory is given, use it. Otherwise, use the current working
 * directory.
 *
 * @param startDir Optional starting directory for the search. If not given, the current directory is used.
 * @returns Path to `package.json`, or `undefined` if not found.
 */
function findPackage(startDir) {
    return (0, path_1.findUp)("package.json", { startDir });
}
/**
 * Find the parent directory of the nearest `package.json` manifest file. Search
 * upward through all parent directories.
 *
 * If a starting directory is given, use it. Otherwise, use the current working
 * directory.
 *
 * @param startDir Optional starting directory for the search. If not given, the current directory is used.
 * @returns Path to `package.json`, or `undefined` if not found.
 */
function findPackageDir(startDir) {
    const manifest = (0, path_1.findUp)("package.json", { startDir });
    return manifest && path.dirname(manifest);
}
/**
 * Find the package dependency's directory, starting from the given directory
 * and moving outward, through all parent directories.
 *
 * Package dependencies exist under 'node_modules/[`scope`]/[`name`]'.
 *
 * @param ref Package dependency reference
 * @param options Options which control the search
 * @returns Path to the package dependency's directory, or `undefined` if not found.
 */
function findPackageDependencyDir(ref, options) {
    var _a;
    const pkgName = typeof ref === "string" ? ref : path.join((_a = ref.scope) !== null && _a !== void 0 ? _a : "", ref.name);
    const packageDir = (0, path_1.findUp)(path.join("node_modules", pkgName), {
        startDir: options === null || options === void 0 ? void 0 : options.startDir,
        type: "directory",
        allowSymlinks: options === null || options === void 0 ? void 0 : options.allowSymlinks,
    });
    if (!packageDir || !(options === null || options === void 0 ? void 0 : options.resolveSymlinks)) {
        return packageDir;
    }
    return fs.lstatSync(packageDir).isSymbolicLink()
        ? path.resolve(path.dirname(packageDir), fs.readlinkSync(packageDir))
        : packageDir;
}
/**
 * Resolve the path to a dependency given a chain of dependencies leading up to
 * it.
 * @param chain Chain of dependencies leading up to the target dependency.
 * @param startDir Optional starting directory for the search. If not given, the current directory is used.
 * @returns Path to the final dependency's directory.
 */
function resolveDependencyChain(chain, startDir = process.cwd()) {
    return chain.reduce((startDir, module) => {
        const p = require.resolve(`${module}/package.json`, { paths: [startDir] });
        return path.dirname(p);
    }, startDir);
}
//# sourceMappingURL=package.js.map