"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    getAnswerBody: function() {
        return getAnswerBody;
    },
    getAutocompleteComponentMap: function() {
        return getAutocompleteComponentMap;
    },
    takeMostRecentAnswers: function() {
        return takeMostRecentAnswers;
    },
    transformAnswers: function() {
        return transformAnswers;
    }
});
const _datefns = require("date-fns");
const _lodash = require("lodash");
const _dateTime = require("@tamanu/utils/dateTime");
const _constants = require("@tamanu/constants");
// also update getNameColumnForModel in /packages/mobile/App/ui/helpers/fields.ts when this changes
function getNameColumnForModel(modelName) {
    switch(modelName){
        case 'User':
            return 'displayName';
        default:
            return 'name';
    }
}
// also update getDisplayNameForModel in /packages/mobile/App/ui/helpers/fields.ts when this changes
function getDisplayNameForModel(modelName, record) {
    const columnName = getNameColumnForModel(modelName);
    return record[columnName] || record.id;
}
const convertAutocompleteAnswer = async (models, componentConfig, answer)=>{
    if (!componentConfig) {
        return answer;
    }
    const model = models[componentConfig.source];
    if (!model) {
        throw new Error(`no model for componentConfig ${JSON.stringify(componentConfig)}`);
    }
    const result = await model.findByPk(answer);
    if (!result) {
        if (answer === '') {
            return answer;
        }
        if (componentConfig.source === 'ReferenceData') {
            throw new Error(`Selected answer ${componentConfig.source}[${answer}] not found (check that the surveyquestion's source isn't ReferenceData for a Location, Facility, or Department)`);
        }
        throw new Error(`Selected answer ${componentConfig.source}[${answer}] not found`);
    }
    return getDisplayNameForModel(componentConfig.source, result);
};
const convertBinaryToYesNo = (answer)=>{
    switch(answer){
        case 'true':
        case '1':
            return 'Yes';
        case 'false':
        case '0':
            return 'No';
        default:
            return answer;
    }
};
const convertDateAnswer = (answer, { dateFormat = 'dd-MM-yyyy', notTransformDate = false })=>{
    if (notTransformDate) return answer;
    if ((0, _dateTime.isISOString)(answer)) {
        return (0, _dateTime.format)(answer, dateFormat);
    }
    return '';
};
// Logic duplicated in packages/mobile/App/ui/navigation/screens/programs/SurveyResponseDetailsScreen/index.tsx
const isAutocomplete = ({ config, dataElement })=>dataElement?.type === _constants.PROGRAM_DATA_ELEMENT_TYPES.AUTOCOMPLETE || config && JSON.parse(config).writeToPatient?.fieldType === _constants.PROGRAM_DATA_ELEMENT_TYPES.AUTOCOMPLETE;
const getAnswerBody = async (models, componentConfig, type, answer, transformConfig)=>{
    switch(type){
        case 'Date':
        case 'SubmissionDate':
            return convertDateAnswer(answer, transformConfig);
        case 'Checkbox':
            return convertBinaryToYesNo(answer);
        case 'Autocomplete':
            return convertAutocompleteAnswer(models, componentConfig, answer);
        default:
            {
                if (isAutocomplete({
                    config: JSON.stringify(componentConfig)
                })) {
                    return convertAutocompleteAnswer(models, componentConfig, answer);
                }
                return answer;
            }
    }
};
const getAutocompleteComponentMap = async (models, surveyComponents)=>{
    // Get all survey answer components with config and type from source
    const surveyAnswerComponents = await Promise.all(surveyComponents.filter((c)=>c.dataElement.type === _constants.PROGRAM_DATA_ELEMENT_TYPES.SURVEY_ANSWER).map(async (c)=>{
        const config = JSON.parse(c.config);
        const ssc = await models.SurveyScreenComponent.findOne({
            include: [
                {
                    model: models.ProgramDataElement,
                    as: 'dataElement',
                    where: {
                        code: config.source || config.Source
                    }
                }
            ]
        });
        return {
            ...c,
            lookupSurveyScreenComponent: ssc
        };
    }));
    const surveyAnswerAutocompleteComponents = surveyAnswerComponents.filter((c)=>isAutocomplete(c.lookupSurveyScreenComponent)).map((c)=>[
            c.dataElementId,
            c.lookupSurveyScreenComponent.config ? JSON.parse(c.lookupSurveyScreenComponent.config) : {}
        ]);
    const autocompleteComponents = surveyComponents.filter(isAutocomplete).map(({ dataElementId, config: componentConfig })=>[
            dataElementId,
            componentConfig ? JSON.parse(componentConfig) : {}
        ]);
    return new Map([
        ...autocompleteComponents,
        ...surveyAnswerAutocompleteComponents
    ]);
};
const transformAnswers = async (models, surveyResponseAnswers, surveyComponents, transformConfig = {})=>{
    const autocompleteComponentMap = await getAutocompleteComponentMap(models, surveyComponents);
    const dataElementIdToComponent = (0, _lodash.keyBy)(surveyComponents, (component)=>component.dataElementId);
    // Some questions in the front end are not answered but still record the answer as empty string in the database
    // So we should filter any answers that are empty.
    const nonEmptyAnswers = surveyResponseAnswers.filter((answer)=>answer.body !== null && answer.body !== undefined && answer.body !== '');
    const transformedAnswers = [];
    // Transform Autocomplete answers from: ReferenceData.id to ReferenceData.name
    for (const answer of nonEmptyAnswers){
        const surveyId = answer.surveyResponse?.surveyId;
        const surveyResponseId = answer.surveyResponse?.id;
        const patientId = answer.surveyResponse?.encounter?.patientId;
        const responseEndTime = answer.surveyResponse?.endTime;
        const { dataElementId } = answer;
        let type = dataElementIdToComponent[dataElementId]?.dataElement?.dataValues?.type;
        let sourceType;
        const componentConfig = autocompleteComponentMap.get(dataElementId);
        // If the answer is a survey answer, we need to look up the type from the source survey components
        if (type === _constants.PROGRAM_DATA_ELEMENT_TYPES.SURVEY_ANSWER) {
            const config = JSON.parse(dataElementIdToComponent[dataElementId].config);
            const ssc = await models.SurveyScreenComponent.findOne({
                include: [
                    {
                        model: models.ProgramDataElement,
                        as: 'dataElement',
                        where: {
                            code: config.source || config.Source
                        }
                    }
                ]
            });
            sourceType = ssc.dataElement.dataValues.type;
        }
        const body = await getAnswerBody(models, componentConfig, sourceType || type, answer.body, transformConfig);
        const answerObject = {
            id: answer.id,
            surveyId,
            surveyResponseId,
            patientId,
            responseEndTime,
            dataElementId,
            body,
            sourceType
        };
        transformedAnswers.push(answerObject);
    }
    return transformedAnswers;
};
const takeMostRecentAnswers = (answers)=>{
    const answersPerElement = (0, _lodash.groupBy)(answers, (a)=>`${a.patientId}|${a.surveyId}|${a.dataElementId}`);
    const results = [];
    for (const groupedAnswers of Object.values(answersPerElement)){
        const sortedLatestToOldestAnswers = groupedAnswers.sort((a1, a2)=>(0, _dateTime.differenceInMilliseconds)((0, _datefns.parseISO)(a2.responseEndTime), (0, _datefns.parseISO)(a1.responseEndTime)));
        results.push(sortedLatestToOldestAnswers[0]);
    }
    return results;
};

//# sourceMappingURL=transformAnswers.js.map