"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    ApplicationContext: function() {
        return ApplicationContext;
    },
    CENTRAL_SERVER_APP_TYPES: function() {
        return CENTRAL_SERVER_APP_TYPES;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _lodash = require("lodash");
const _settings = require("@tamanu/settings");
const _dataMigrations = require("@tamanu/database/dataMigrations");
const _logging = require("@tamanu/shared/services/logging");
const _EmailService = require("./services/EmailService");
const _database = require("./database");
const _integrations = require("./integrations");
const _TelegramBotService = require("./services/TelegramBotService");
const _versionCompatibility = require("./middleware/versionCompatibility");
function _define_property(obj, key, value) {
    if (key in obj) {
        Object.defineProperty(obj, key, {
            value: value,
            enumerable: true,
            configurable: true,
            writable: true
        });
    } else {
        obj[key] = value;
    }
    return obj;
}
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const CENTRAL_SERVER_APP_TYPES = {
    API: 'api',
    FHIR_WORKER: 'fhir-worker',
    MAIN: 'main',
    MIGRATE: 'migrate',
    TASKS: 'tasks'
};
let ApplicationContext = class ApplicationContext {
    async init({ testMode, appType = CENTRAL_SERVER_APP_TYPES.MAIN, dbKey } = {}) {
        if (_config.default.errors?.enabled) {
            if (_config.default.errors.type === 'bugsnag') {
                await (0, _logging.initBugsnag)({
                    ...(0, _lodash.omit)(_config.default.errors, [
                        'enabled',
                        'type'
                    ]),
                    appVersion: [
                        _versionCompatibility.VERSION,
                        process.env.REVISION
                    ].filter(Boolean).join('-'),
                    appType
                });
            }
        }
        this.store = await (0, _database.initDatabase)({
            testMode,
            dbKey: dbKey ?? appType
        });
        this.closePromise = new Promise((resolve)=>{
            this.onClose(resolve);
        });
        this.settings = new _settings.ReadSettings(this.store.models);
        // no need to set up services, integrations, etc. for migrations
        if (appType === CENTRAL_SERVER_APP_TYPES.MIGRATE) {
            return this;
        }
        this.emailService = new _EmailService.EmailService();
        if (_config.default.db.reportSchemas?.enabled) {
            this.reportSchemaStores = await (0, _database.initReporting)();
        }
        this.telegramBotService = await (0, _TelegramBotService.defineSingletonTelegramBotService)({
            config: _config.default,
            models: this.store.models
        });
        if (await (0, _dataMigrations.isSyncTriggerDisabled)(this.store.sequelize)) {
            _logging.log.warn('Sync Trigger is disabled in the database.');
            return null;
        }
        await (0, _integrations.initIntegrations)(this);
        return this;
    }
    onClose(hook) {
        this.closeHooks.push(hook);
    }
    async close() {
        for (const hook of this.closeHooks){
            await hook();
        }
        await (0, _database.closeDatabase)();
    }
    async waitForClose() {
        return this.closePromise;
    }
    constructor(){
        /** @type {Awaited<ReturnType<typeof initDatabase>>|null} */ _define_property(this, "store", null);
        _define_property(this, "reportSchemaStores", null);
        /** @type {EmailService | null} */ _define_property(this, "emailService", null);
        /** @type {Awaited<ReturnType<typeof defineSingletonTelegramBotService>>|null} */ _define_property(this, "telegramBotService", null);
        _define_property(this, "integrations", null);
        /**@type {ReadSettings<CentralSettingPath> | null} */ _define_property(this, "settings", null);
        _define_property(this, "closeHooks", []);
    }
};

//# sourceMappingURL=ApplicationContext.js.map