"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "TamanuApi", {
    enumerable: true,
    get: function() {
        return TamanuApi;
    }
});
const _qs = /*#__PURE__*/ _interop_require_default(require("qs"));
const _constants = require("@tamanu/constants");
const _errors = require("@tamanu/shared/errors");
const _buildAbility = require("@tamanu/shared/permissions/buildAbility");
const _errors1 = require("./errors");
const _fetch = require("./fetch");
const _InterceptorManager = require("./InterceptorManager");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let TamanuApi = class TamanuApi {
    #host;
    #prefix;
    #onAuthFailure;
    #onVersionIncompatible;
    #authHeader;
    lastRefreshed = null;
    user = null;
    constructor({ endpoint, agentName, agentVersion, deviceId }){
        this.#prefix = endpoint;
        const endpointUrl = new URL(endpoint);
        this.#host = endpointUrl.origin;
        this.agentName = agentName;
        this.agentVersion = agentVersion;
        this.deviceId = deviceId;
        this.interceptors = {
            request: new _InterceptorManager.InterceptorManager(),
            response: new _InterceptorManager.InterceptorManager()
        };
    }
    getHost() {
        return this.#host;
    }
    setAuthFailureHandler(handler) {
        this.#onAuthFailure = handler;
    }
    setVersionIncompatibleHandler(handler) {
        this.#onVersionIncompatible = handler;
    }
    async login(email, password) {
        const response = await this.post('login', {
            email,
            password,
            deviceId: this.deviceId
        }, {
            returnResponse: true
        });
        const serverType = response.headers.get('X-Tamanu-Server');
        if (![
            _constants.SERVER_TYPES.FACILITY,
            _constants.SERVER_TYPES.CENTRAL
        ].includes(serverType)) {
            throw new Error(`Tamanu server type '${serverType}' is not supported.`);
        }
        const { token, localisation, server = {}, availableFacilities, permissions, centralHost, role } = await response.json();
        server.type = serverType;
        server.centralHost = centralHost;
        this.setToken(token);
        const { user, ability } = await this.fetchUserData(permissions);
        return {
            user,
            token,
            localisation,
            server,
            availableFacilities,
            ability,
            role,
            permissions
        };
    }
    async fetchUserData(permissions) {
        const user = await this.get('user/me');
        this.lastRefreshed = Date.now();
        this.user = user;
        const ability = (0, _buildAbility.buildAbilityForUser)(user, permissions);
        return {
            user,
            ability
        };
    }
    async requestPasswordReset(email) {
        return this.post('resetPassword', {
            email
        });
    }
    async changePassword(args) {
        return this.post('changePassword', args);
    }
    async refreshToken() {
        try {
            const response = await this.post('refresh');
            const { token } = response;
            this.setToken(token);
        } catch (e) {
            // eslint-disable-next-line no-console
            console.error(e);
        }
    }
    setToken(token) {
        this.#authHeader = {
            authorization: `Bearer ${token}`
        };
    }
    async fetch(endpoint, query = {}, moreConfig = {}) {
        const { headers, returnResponse = false, throwResponse = false, ...otherConfig } = moreConfig;
        const queryString = _qs.default.stringify(query || {});
        const path = `${endpoint}${query ? `?${queryString}` : ''}`;
        const url = `${this.#prefix}/${path}`;
        const config = {
            headers: {
                ...this.#authHeader,
                ...headers,
                'X-Tamanu-Client': this.agentName,
                'X-Version': this.agentVersion
            },
            ...otherConfig
        };
        const requestInterceptorChain = [];
        // request: first in last out
        this.interceptors.request.forEach(function unshiftRequestInterceptors(interceptor) {
            requestInterceptorChain.unshift(interceptor.fulfilled, interceptor.rejected);
        });
        let i = 0;
        let requestPromise = Promise.resolve(config);
        while(i < requestInterceptorChain.length){
            requestPromise = requestPromise.then(requestInterceptorChain[i++], requestInterceptorChain[i++]);
        }
        const latestConfig = await requestPromise;
        const response = await (0, _fetch.fetchOrThrowIfUnavailable)(url, latestConfig);
        const responseInterceptorChain = [];
        // request: first in first out
        this.interceptors.response.forEach(function pushResponseInterceptors(interceptor) {
            responseInterceptorChain.push(interceptor.fulfilled, interceptor.rejected);
        });
        let j = 0;
        let responsePromise = response.ok ? Promise.resolve(response) : Promise.reject(response);
        while(j < responseInterceptorChain.length){
            responsePromise = responsePromise.then(responseInterceptorChain[j++], responseInterceptorChain[j++]);
        }
        await responsePromise.catch(()=>{});
        if (response.ok) {
            if (returnResponse) {
                return response;
            }
            if (response.status === 204) {
                return null;
            }
            return response.json();
        }
        if (throwResponse) {
            throw response;
        }
        return this.extractError(endpoint, response);
    }
    /**
   * Handle errors from the server response.
   *
   * Generally only used internally.
   */ async extractError(endpoint, response) {
        const { error } = await (0, _fetch.getResponseErrorSafely)(response);
        const message = error?.message || response.status.toString();
        // handle forbidden error and trigger catch all modal
        if (response.status === 403 && error) {
            throw new _errors.ForbiddenError(message);
        }
        if (response.status === 404) {
            throw new _errors.NotFoundError(message);
        }
        // handle auth expiring
        if (response.status === 401 && endpoint !== 'login' && this.#onAuthFailure) {
            const message = 'Your session has expired. Please log in again.';
            this.#onAuthFailure(message);
            throw new _errors1.AuthExpiredError(message);
        }
        // handle version incompatibility
        if (response.status === 400 && error) {
            const versionIncompatibleMessage = (0, _errors1.getVersionIncompatibleMessage)(error, response);
            if (versionIncompatibleMessage) {
                if (this.#onVersionIncompatible) {
                    this.#onVersionIncompatible(versionIncompatibleMessage);
                }
                throw new _errors1.VersionIncompatibleError(versionIncompatibleMessage);
            }
        }
        // Handle resource conflict
        if (response.status === 409) {
            throw new _errors1.ResourceConflictError(message);
        }
        throw new _errors1.ServerResponseError(`Server error response: ${message}`);
    }
    async get(endpoint, query = {}, config = {}) {
        return this.fetch(endpoint, query, {
            ...config,
            method: 'GET'
        });
    }
    async download(endpoint, query = {}) {
        const response = await this.fetch(endpoint, query, {
            returnResponse: true
        });
        const blob = await response.blob();
        return blob;
    }
    async postWithFileUpload(endpoint, file, body, options = {}) {
        const blob = new Blob([
            file
        ]);
        // We have to use multipart/formdata to support sending the file data,
        // but sending the other fields in that format loses type information
        // (for eg, sending a value of false will arrive as the string "false")
        // So, we just piggyback a json string over the multipart format, and
        // parse that on the backend.
        const formData = new FormData();
        formData.append('jsonData', JSON.stringify(body));
        formData.append('file', blob);
        return this.fetch(endpoint, undefined, {
            method: 'POST',
            body: formData,
            ...options
        });
    }
    async post(endpoint, body = undefined, config = {}) {
        return this.fetch(endpoint, {}, {
            body: body && JSON.stringify(body),
            headers: {
                'Content-Type': 'application/json'
            },
            ...config,
            method: 'POST'
        });
    }
    async put(endpoint, body = undefined, config = {}) {
        return this.fetch(endpoint, {}, {
            body: body && JSON.stringify(body),
            headers: {
                'Content-Type': 'application/json'
            },
            ...config,
            method: 'PUT'
        });
    }
    async delete(endpoint, query = {}, config = {}) {
        return this.fetch(endpoint, query, {
            ...config,
            method: 'DELETE'
        });
    }
};

//# sourceMappingURL=TamanuApi.js.map