"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "programRegistry", {
    enumerable: true,
    get: function() {
        return programRegistry;
    }
});
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _sequelize = require("sequelize");
const _ability = require("@casl/ability");
const _constants = require("@tamanu/constants");
const _renameObjectKeys = require("@tamanu/utils/renameObjectKeys");
const _crudHelpers = require("@tamanu/shared/utils/crudHelpers");
const _query = require("../../utils/query");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const programRegistry = _express.default.Router();
programRegistry.get('/:id', (0, _crudHelpers.simpleGet)('ProgramRegistry'));
programRegistry.get('/$', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, query } = req;
    req.checkPermission('list', 'ProgramRegistry');
    if (query.excludePatientId) {
        req.checkPermission('read', 'Patient');
        req.checkPermission('read', 'PatientProgramRegistration');
    }
    const { ProgramRegistry } = models;
    const patientIdExclusion = query.excludePatientId ? {
        id: {
            [_sequelize.Op.notIn]: _sequelize.Sequelize.literal(`(
                SELECT most_recent_registrations.id
                FROM (
                    SELECT DISTINCT ON (pr.id) pr.id, ppr.registration_status
                    from program_registries pr
                    INNER JOIN patient_program_registrations ppr
                    ON ppr.program_registry_id = pr.id
                    WHERE ppr.patient_id = :excludePatientId
                    ORDER BY pr.id DESC, ppr.date DESC, ppr.id DESC
                ) most_recent_registrations
                WHERE most_recent_registrations.registration_status != :error
              )`)
        }
    } : {};
    const baseQueryOptions = {
        where: {
            visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT,
            ...patientIdExclusion
        },
        replacements: {
            error: _constants.REGISTRATION_STATUSES.RECORDED_IN_ERROR,
            excludePatientId: query.excludePatientId
        }
    };
    const count = await ProgramRegistry.count(baseQueryOptions);
    const { order = 'ASC', orderBy = 'createdAt', rowsPerPage, page } = query;
    const objects = await ProgramRegistry.findAll({
        ...baseQueryOptions,
        include: ProgramRegistry.getListReferenceAssociations(models),
        order: orderBy ? [
            [
                ...orderBy.split('.'),
                order.toUpperCase()
            ]
        ] : undefined,
        limit: rowsPerPage,
        offset: page && rowsPerPage ? page * rowsPerPage : undefined
    });
    const filteredObjects = objects.filter((programRegistry)=>req.ability.can('list', programRegistry));
    const filteredData = filteredObjects.map((x)=>x.forResponse());
    const filteredCount = objects.length !== filteredObjects.length ? filteredObjects.length : count;
    res.send({
        count: filteredCount,
        data: filteredData
    });
}));
programRegistry.get('/:id/conditions', (0, _crudHelpers.simpleGetList)('ProgramRegistryCondition', 'programRegistryId', {
    additionalFilters: {
        visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT
    }
}));
programRegistry.get('/:id/registrations', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models: { PatientProgramRegistration }, params: { id: programRegistryId }, query } = req;
    req.checkPermission('read', (0, _ability.subject)('ProgramRegistry', {
        id: programRegistryId
    }));
    req.checkPermission('read', 'Patient');
    req.checkPermission('list', 'PatientProgramRegistration');
    const { order = 'ASC', orderBy = 'displayId', rowsPerPage = 10, page = 0, ...filterParams } = query;
    const makeSimpleTextFilter = (0, _query.makeSimpleTextFilterFactory)(filterParams);
    const makePartialTextFilter = (0, _query.makeSubstringTextFilterFactory)(filterParams);
    const filters = [
        // Patient filters
        makePartialTextFilter('displayId', 'patient.display_id'),
        makeSimpleTextFilter('firstName', 'patient.first_name'),
        makeSimpleTextFilter('lastName', 'patient.last_name'),
        (0, _query.makeFilter)(filterParams.sex, 'patient.sex = :sex', ({ sex })=>({
                sex: sex.toLowerCase()
            })),
        (0, _query.makeFilter)(filterParams.dateOfBirth, `patient.date_of_birth = :dateOfBirth`),
        (0, _query.makeFilter)(filterParams.homeVillage, `patient.village_id = :homeVillage`),
        (0, _query.makeFilter)(filterParams.divisionId, `pad.division_id = :divisionId`),
        (0, _query.makeFilter)(filterParams.subdivisionId, `pad.subdivision_id = :subdivisionId`),
        (0, _query.makeFilter)(!filterParams.deceased || filterParams.deceased === 'false', 'patient.date_of_death IS NULL'),
        // Registration filters
        (0, _query.makeFilter)(filterParams.registeringFacilityId, 'mrr.registering_facility_id = :registeringFacilityId'),
        (0, _query.makeFilter)(filterParams.clinicalStatus, 'mrr.clinical_status_id = any(array[:clinicalStatus])'),
        (0, _query.makeFilter)(filterParams.currentlyIn, 'mrr.village_id = :currentlyIn OR mrr.facility_id = :currentlyIn'),
        (0, _query.makeFilter)(filterParams.programRegistryCondition, // Essentially the `<@` operator checks that the json on the left is contained in the json on the right
        // so we build up a string like '["A_condition_name"]' and cast it to json before checking membership.
        `(select array_agg(prc2.name) from program_registry_conditions prc2 where prc2.id = any(array[:programRegistryCondition])) && conditions.condition_list`),
        (0, _query.makeFilter)(true, 'mrr.registration_status != :error_status', ()=>({
                error_status: _constants.REGISTRATION_STATUSES.RECORDED_IN_ERROR
            })),
        (0, _query.makeFilter)(!filterParams.removed || filterParams.removed === 'false', 'mrr.registration_status = :active_status', ()=>({
                active_status: _constants.REGISTRATION_STATUSES.ACTIVE
            }))
    ].filter((f)=>f);
    const whereClauses = filters.map((f)=>f.sql).join(' AND ');
    const filterReplacements = filters.filter((f)=>f.transform).reduce((current, { transform })=>({
            ...current,
            ...transform(current)
        }), filterParams);
    const withClause = `
      with
        most_recent_registrations as (
          SELECT *
          FROM (
            SELECT
              *,
              ROW_NUMBER() OVER (PARTITION BY patient_id, program_registry_id ORDER BY date DESC, id DESC) AS row_num
            FROM patient_program_registrations
            WHERE program_registry_id = :programRegistryId
            AND is_most_recent IS TRUE
          ) n
          WHERE n.row_num = 1
        ),
        conditions as (
          SELECT patient_id, array_agg(prc."name") condition_list, jsonb_agg(jsonb_build_object('id', prc.id, 'name', prc."name")) condition_records_list
          FROM patient_program_registration_conditions pprc
            JOIN program_registry_conditions prc
              ON pprc.program_registry_condition_id = prc.id
          WHERE pprc.program_registry_id = :programRegistryId AND pprc.deleted_at IS NULL
          GROUP BY patient_id
        )
    `;
    const from = `
      FROM most_recent_registrations mrr
        LEFT JOIN patients patient
          ON patient.id = mrr.patient_id
        LEFT JOIN reference_data patient_village
          ON patient.village_id = patient_village.id
        LEFT JOIN reference_data currently_at_village
          ON mrr.village_id = currently_at_village.id
        LEFT JOIN facilities currently_at_facility
          ON mrr.facility_id = currently_at_facility.id
        LEFT JOIN facilities registering_facility
          ON mrr.registering_facility_id = registering_facility.id
        LEFT JOIN conditions
          ON conditions.patient_id = mrr.patient_id
        LEFT JOIN program_registry_clinical_statuses status
          ON mrr.clinical_status_id = status.id
        LEFT JOIN program_registries program_registry
          ON mrr.program_registry_id = program_registry.id
        LEFT JOIN users clinician
          ON mrr.clinician_id = clinician.id
        LEFT JOIN patient_additional_data pad
          ON pad.patient_id = patient.id
        LEFT JOIN reference_data division
          ON division.id = pad.division_id
        LEFT JOIN reference_data subdivision
          ON subdivision.id = pad.subdivision_id
      ${whereClauses && `WHERE ${whereClauses}`}
    `;
    const countResult = await req.db.query(`${withClause} SELECT COUNT(1) AS count ${from}`, {
        replacements: {
            ...filterReplacements,
            programRegistryId
        },
        type: _sequelize.QueryTypes.SELECT
    });
    const count = parseInt(countResult[0].count, 10);
    if (count === 0) {
        // save ourselves a query
        res.send({
            data: [],
            count
        });
        return;
    }
    const sortKeys = {
        displayId: 'patient.display_id',
        firstName: 'UPPER(patient.first_name)',
        lastName: 'UPPER(patient.last_name)',
        dateOfBirth: 'patient.date_of_birth',
        homeVillage: 'UPPER(patient_village.name)',
        registeringFacility: 'registering_facility.name',
        currentlyIn: 'COALESCE(UPPER(currently_at_village.name), UPPER(currently_at_facility.name))',
        clinicalStatus: 'mrr.clinical_status_id',
        divisionName: 'patient.division.name',
        subdivisionName: 'patient.subdivision.name'
    };
    const sortKey = sortKeys[orderBy] ?? sortKeys.displayId;
    const sortDirection = order.toLowerCase() === 'asc' ? 'ASC' : 'DESC';
    const nullPosition = sortDirection === 'ASC' ? 'NULLS FIRST' : 'NULLS LAST';
    const result = await req.db.query(`
      ${withClause}
      select
        patient.id AS "patient.id",
        --
        -- Details for the table
        patient.id AS "patient_id",
        patient.display_id AS "patient.display_id",
        patient.first_name AS "patient.first_name",
        patient.last_name AS "patient.last_name",
        patient.date_of_birth AS "patient.date_of_birth",
        patient.date_of_death AS "patient.date_of_death",
        patient.sex AS "patient.sex",
        division.name AS "patient.division.name",
        subdivision.name AS "patient.subdivision.name",
        patient_village.name AS "patient.village.name",
        currently_at_village.name as "village.name",
        currently_at_facility.name as "facility.name",
        registering_facility.name as "registering_facility.name",
        registering_facility.id as "registering_facility_id",
        conditions.condition_records_list as "conditions",
        status.name as "clinical_status.name",
        status.color as "clinical_status.color",
        status.id as "clinical_status.id",
        program_registry.currently_at_type as "program_registry.currently_at_type",
        program_registry.name as "program_registry.name",
        program_registry.id as "program_registry_id",
        clinician.display_name as "clinician.display_name",
        mrr.date as "date",
        --
        -- Details for filtering/ordering
        patient.date_of_death as "patient.date_of_death",
        mrr.registration_status as "registration_status"
      ${from}

      ORDER BY ${sortKey} ${sortDirection}${nullPosition ? ` ${nullPosition}` : ''}
      LIMIT :limit
      OFFSET :offset
      `, {
        replacements: {
            ...filterReplacements,
            programRegistryId,
            limit: rowsPerPage,
            offset: page * rowsPerPage,
            sortKey,
            sortDirection
        },
        // The combination of these two parameters allow mapping the query results
        // to nested models
        model: PatientProgramRegistration,
        mapToModel: true,
        nest: true,
        raw: true,
        type: _sequelize.QueryTypes.SELECT
    });
    const forResponse = result.map(_renameObjectKeys.deepRenameObjectKeys);
    res.send({
        data: forResponse,
        count
    });
}));

//# sourceMappingURL=programRegistry.js.map