"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "SignerRenewalChecker", {
    enumerable: true,
    get: function() {
        return SignerRenewalChecker;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _tasks = require("@tamanu/shared/tasks");
const _logging = require("@tamanu/shared/services/logging");
const _sequelize = require("sequelize");
const _Signer = require("../integrations/Signer");
const _localisation = require("../localisation");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let SignerRenewalChecker = class SignerRenewalChecker extends _tasks.ScheduledTask {
    getName() {
        return 'SignerRenewalChecker';
    }
    async run() {
        const { Signer } = this.context.store.models;
        const pending = await Signer.findAll({
            where: {
                certificate: {
                    [_sequelize.Op.is]: null
                },
                privateKey: {
                    [_sequelize.Op.not]: null
                }
            }
        });
        if (pending.length > 0) {
            _logging.log.info(`There is at least one pending signer CSR: ${pending.map((s)=>s.id).join(', ')}, skipping renewal`);
            return;
        }
        const signer = await Signer.findActive();
        let beyondThreshold = false;
        // If this is the first signer
        if (!signer) {
            beyondThreshold = true;
        } else {
            // If we've let it lapse entirely
            if (signer.validityPeriodEnd <= new Date()) {
                beyondThreshold = true;
            }
            // If we're really too late somehow
            if (signer.workingPeriodEnd <= new Date()) {
                beyondThreshold = true;
            }
            // Buffer before PKUP ends
            const daysUntilWorkingEnd = (signer.workingPeriodEnd - new Date()) / (1000 * 60 * 60 * 24);
            if (daysUntilWorkingEnd <= 16) {
                beyondThreshold = true;
            }
        }
        if (beyondThreshold) {
            if (signer) {
                _logging.log.info(`Signer ${signer.id} is beyond renewal threshold`);
            } else {
                _logging.log.info(`Generating CSR for first Signer`);
            }
            _logging.log.info('Generating new signer CSR');
            const { publicKey, privateKey, request } = await (0, _Signer.newKeypairAndCsr)();
            const newSigner = await Signer.create({
                publicKey: Buffer.from(publicKey),
                privateKey: Buffer.from(privateKey),
                request,
                countryCode: (await (0, _localisation.getLocalisation)()).country['alpha-3']
            });
            _logging.log.info(`Created new signer (CSR): ${newSigner.id}`);
        }
    }
    constructor(context){
        const conf = _config.default.schedules.signerRenewalChecker;
        const { schedule, jitterTime, enabled } = conf;
        super(schedule, _logging.log, jitterTime, enabled);
        this.config = conf;
        this.context = context;
        this.runImmediately();
    }
};

//# sourceMappingURL=SignerRenewalChecker.js.map