"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "Task", {
    enumerable: true,
    get: function() {
        return Task;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _uuid = require("uuid");
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _ms = /*#__PURE__*/ _interop_require_default(require("ms"));
const _datefns = require("date-fns");
const _dateTime = require("@tamanu/utils/dateTime");
const _Model = require("./Model");
const _buildEncounterLinkedSyncFilter = require("../sync/buildEncounterLinkedSyncFilter");
const _buildEncounterLinkedLookupFilter = require("../sync/buildEncounterLinkedLookupFilter");
const _model = require("../types/model");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const TASK_STATUS_VALUES = Object.values(_constants.TASK_STATUSES);
let Task = class Task extends _Model.Model {
    static initModel({ primaryKey, ...options }, models) {
        super.init({
            id: primaryKey,
            name: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: false
            },
            dueTime: (0, _model.dateTimeType)('dueTime', {
                allowNull: false
            }),
            endTime: (0, _model.dateTimeType)('endTime', {
                allowNull: true
            }),
            requestTime: (0, _model.dateTimeType)('requestTime', {
                allowNull: false
            }),
            status: {
                type: _sequelize.DataTypes.ENUM(...TASK_STATUS_VALUES),
                defaultValue: _constants.TASK_STATUSES.TODO,
                allowNull: false
            },
            note: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: true
            },
            frequencyValue: {
                type: _sequelize.DataTypes.DECIMAL,
                allowNull: true
            },
            frequencyUnit: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            },
            durationValue: {
                type: _sequelize.DataTypes.DECIMAL,
                allowNull: true
            },
            durationUnit: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            },
            highPriority: {
                type: _sequelize.DataTypes.BOOLEAN,
                allowNull: true
            },
            parentTaskId: {
                type: _sequelize.DataTypes.UUID,
                allowNull: true,
                references: {
                    model: this,
                    key: 'id'
                }
            },
            completedTime: (0, _model.dateTimeType)('completedTime', {
                allowNull: true
            }),
            completedNote: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: true
            },
            notCompletedTime: (0, _model.dateTimeType)('notCompletedTime', {
                allowNull: true
            }),
            todoTime: (0, _model.dateTimeType)('todoTime', {
                allowNull: true
            }),
            todoNote: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: true
            },
            deletedTime: (0, _model.dateTimeType)('deletedTime', {
                allowNull: true
            })
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL,
            hooks: {
                async afterDestroy (task, opts) {
                    await models.TaskDesignation.destroy({
                        where: {
                            taskId: task.id
                        },
                        transaction: opts.transaction
                    });
                }
            }
        });
    }
    isRepeatingTask() {
        return this.frequencyValue && this.frequencyUnit;
    }
    static initRelations(models) {
        this.belongsTo(models.Task, {
            foreignKey: 'parentTaskId',
            as: 'parentTask'
        });
        this.belongsTo(models.Encounter, {
            foreignKey: 'encounterId',
            as: 'encounter'
        });
        this.belongsTo(models.User, {
            foreignKey: 'requestedByUserId',
            as: 'requestedBy'
        });
        this.belongsTo(models.User, {
            foreignKey: 'completedByUserId',
            as: 'completedBy'
        });
        this.belongsTo(models.User, {
            foreignKey: 'notCompletedByUserId',
            as: 'notCompletedBy'
        });
        this.belongsTo(models.User, {
            foreignKey: 'todoByUserId',
            as: 'todoBy'
        });
        this.belongsTo(models.User, {
            foreignKey: 'deletedByUserId',
            as: 'deletedBy'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'notCompletedReasonId',
            as: 'notCompletedReason'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'deletedReasonId',
            as: 'deletedReason'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'deletedReasonForSyncId',
            as: 'deletedReasonForSync'
        });
        this.belongsToMany(models.ReferenceData, {
            through: models.TaskDesignation,
            foreignKey: 'taskId',
            as: 'designations'
        });
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilter)([
            this.tableName,
            'encounters'
        ], markedForSyncPatientsTable);
    }
    static buildSyncLookupQueryDetails() {
        return (0, _buildEncounterLinkedLookupFilter.buildEncounterLinkedLookupFilter)(this);
    }
    static getFullReferenceAssociations() {
        const { models } = this.sequelize;
        return [
            {
                model: models.Encounter,
                as: 'encounter',
                attributes: [
                    'id'
                ]
            },
            {
                model: models.User,
                as: 'requestedBy',
                attributes: [
                    'displayName'
                ]
            },
            {
                model: models.User,
                as: 'completedBy',
                attributes: [
                    'displayName'
                ]
            },
            {
                model: models.User,
                as: 'notCompletedBy',
                attributes: [
                    'displayName'
                ]
            },
            {
                model: models.User,
                as: 'todoBy',
                attributes: [
                    'displayName'
                ]
            },
            {
                model: models.User,
                as: 'deletedBy',
                attributes: [
                    'displayName'
                ]
            },
            {
                model: models.ReferenceData,
                as: 'notCompletedReason',
                attributes: [
                    'name'
                ]
            },
            {
                model: models.ReferenceData,
                as: 'deletedReason',
                attributes: [
                    'name'
                ]
            },
            {
                model: models.ReferenceData,
                as: 'designations',
                attributes: [
                    'id',
                    'name'
                ],
                through: {
                    attributes: []
                }
            }
        ];
    }
    static async generateRepeatingTasks(tasks) {
        const allGeneratedTasks = [];
        const allClonedDesignations = [];
        const repeatingTasks = tasks.filter((task)=>task.frequencyValue && task.frequencyUnit);
        for (const task of repeatingTasks){
            let lastGeneratedTask = await this.findOne({
                where: {
                    parentTaskId: task.id
                },
                order: [
                    [
                        'dueTime',
                        'DESC'
                    ]
                ]
            });
            if (!lastGeneratedTask) {
                // no tasks have been generated yet
                lastGeneratedTask = task;
            }
            const upcomingTasksShouldBeGeneratedTimeFrame = _config.default.tasking?.upcomingTasksShouldBeGeneratedTimeFrame || 72;
            const { frequencyValue, frequencyUnit, durationValue, durationUnit } = task;
            const frequency = (0, _ms.default)(`${frequencyValue} ${frequencyUnit}`);
            const maxDueTime = (0, _datefns.addMilliseconds)(new Date(), (0, _ms.default)(`${upcomingTasksShouldBeGeneratedTimeFrame} hours`));
            let nextDueTime = (0, _datefns.addMilliseconds)(new Date(lastGeneratedTask.dueTime), frequency);
            const generatedTasks = [];
            // const parsedStartDate = parseISO(startDate);
            // const duration = parseInt(durationValue, 10);
            // add(parsedStartDate, { [durationUnit]: duration });
            let endDate = Infinity;
            if (durationValue && durationUnit) {
                switch(durationUnit){
                    case _constants.TASK_DURATION_UNIT.OCCURRENCES:
                        endDate = (0, _datefns.addMilliseconds)(new Date(task.dueTime), frequency * (durationValue - 1)).getTime();
                        break;
                    default:
                        {
                            const duration = (0, _ms.default)(`${durationValue} ${durationUnit}`);
                            endDate = (0, _datefns.addMilliseconds)(new Date(task.dueTime), duration).getTime();
                            break;
                        }
                }
            }
            // Use a while loop to generate tasks until we reach maxDueTime or endDateTime
            while(nextDueTime.getTime() <= maxDueTime.getTime() && nextDueTime.getTime() <= endDate){
                const nextTask = {
                    id: (0, _uuid.v4)(),
                    encounterId: task.encounterId,
                    requestedByUserId: task.requestedByUserId,
                    name: task.name,
                    dueTime: (0, _dateTime.toDateTimeString)(nextDueTime),
                    requestTime: task.requestTime,
                    status: _constants.TASK_STATUSES.TODO,
                    note: task.note,
                    frequencyValue: task.frequencyValue,
                    frequencyUnit: task.frequencyUnit,
                    durationValue: task.durationValue,
                    durationUnit: task.durationUnit,
                    highPriority: task.highPriority,
                    parentTaskId: task.id
                };
                generatedTasks.push(nextTask);
                // Increment nextDueTime for the next iteration
                nextDueTime = (0, _datefns.addMilliseconds)(nextDueTime, frequency);
            }
            const clonedDesignations = [];
            for (const generatedTask of generatedTasks){
                clonedDesignations.push(...task.designations.map((designation)=>({
                        taskId: generatedTask.id,
                        designationId: designation.id
                    })));
            }
            allGeneratedTasks.push(...generatedTasks);
            allClonedDesignations.push(...clonedDesignations);
        }
        if (allGeneratedTasks.length) {
            await this.bulkCreate(allGeneratedTasks);
        }
        if (allClonedDesignations.length) {
            await this.sequelize.models.TaskDesignation.bulkCreate(allClonedDesignations);
        }
    }
    static async onEncounterDischarged(encounter, transaction) {
        const { models } = this.sequelize;
        const encounterId = encounter.id;
        const endTime = encounter.endDate;
        const taskDeletionReason = await models.ReferenceData.findByPk(_constants.TASK_DELETE_PATIENT_DISCHARGED_REASON_ID);
        await models.Task.update({
            endTime,
            deletedReasonForSyncId: taskDeletionReason?.id ?? null
        }, {
            where: {
                encounterId,
                parentTaskId: null,
                frequencyValue: {
                    [_sequelize.Op.not]: null
                },
                frequencyUnit: {
                    [_sequelize.Op.not]: null
                }
            },
            transaction
        });
        await models.Task.update({
            deletedByUserId: _constants.SYSTEM_USER_UUID,
            deletedReasonId: taskDeletionReason?.id ?? null,
            deletedTime: (0, _dateTime.getCurrentDateTimeString)()
        }, {
            where: {
                encounterId,
                status: _constants.TASK_STATUSES.TODO,
                dueTime: {
                    [_sequelize.Op.gt]: endTime
                },
                frequencyValue: {
                    [_sequelize.Op.not]: null
                },
                frequencyUnit: {
                    [_sequelize.Op.not]: null
                }
            },
            transaction
        });
        await models.Task.destroy({
            where: {
                encounterId,
                status: _constants.TASK_STATUSES.TODO,
                dueTime: {
                    [_sequelize.Op.gt]: endTime
                },
                frequencyValue: {
                    [_sequelize.Op.not]: null
                },
                frequencyUnit: {
                    [_sequelize.Op.not]: null
                }
            },
            transaction,
            individualHooks: true
        });
    }
};

//# sourceMappingURL=Task.js.map