"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "patientDocumentMetadataRoutes", {
    enumerable: true,
    get: function() {
        return patientDocumentMetadataRoutes;
    }
});
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _dateTime = require("@tamanu/utils/dateTime");
const _errors = require("@tamanu/shared/errors");
const _uploadAttachment = require("../../../utils/uploadAttachment");
const _utils = require("../../../database/utils");
const _createPatientLetter = require("../../../routeHandlers/createPatientLetter");
const _deleteModel = require("../../../routeHandlers/deleteModel");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
// Object used to map field names to database column names
const SNAKE_CASE_COLUMN_NAMES = {
    type: 'type',
    documentOwner: 'document_owner',
    name: 'department.name'
};
// Filtering functions for sequelize queries
const caseInsensitiveFilter = (0, _utils.getCaseInsensitiveFilter)(SNAKE_CASE_COLUMN_NAMES);
const patientDocumentMetadataRoutes = _express.default.Router();
// Route used on DocumentsTable component
patientDocumentMetadataRoutes.get('/:id/documentMetadata', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('list', 'DocumentMetadata');
    req.checkPermission('list', 'Encounter');
    const { models, params, query } = req;
    const { order = 'ASC', orderBy, rowsPerPage = 10, page = 0, offset, ...filterParams } = query;
    // Get all encounter IDs for this patient
    const patientId = params.id;
    const patientEncounters = await models.Encounter.findAll({
        where: {
            patientId
        },
        attributes: [
            'id'
        ]
    });
    // Convert into an array of strings for querying
    const encounterIds = patientEncounters.map((obj)=>obj.id);
    // Create filters
    const documentFilters = (0, _utils.mapQueryFilters)(filterParams, [
        {
            key: 'type',
            operator: _sequelize.Op.substring,
            mapFn: caseInsensitiveFilter
        },
        {
            key: 'documentOwner',
            operator: _sequelize.Op.startsWith,
            mapFn: caseInsensitiveFilter
        }
    ]);
    const departmentFilters = (0, _utils.mapQueryFilters)(filterParams, [
        {
            key: 'departmentName',
            alias: 'name',
            operator: _sequelize.Op.startsWith,
            mapFn: caseInsensitiveFilter
        }
    ]);
    // Require it when search has field, otherwise documents
    // without a specified department won't appear
    const departmentInclude = {
        association: 'department',
        where: departmentFilters,
        required: filterParams.departmentName && true || false
    };
    // Get all document metadata associated with the patient or any encounter
    // that the patient may have had. Also apply filters from search bar.
    const documentMetadataItems = await models.DocumentMetadata.findAndCountAll({
        where: {
            [_sequelize.Op.and]: [
                {
                    [_sequelize.Op.or]: [
                        {
                            patientId
                        },
                        {
                            encounterId: {
                                [_sequelize.Op.in]: encounterIds
                            }
                        }
                    ]
                },
                documentFilters
            ]
        },
        order: orderBy ? (0, _utils.getOrderClause)(order, orderBy) : undefined,
        include: [
            departmentInclude
        ],
        limit: rowsPerPage,
        offset: offset || page * rowsPerPage
    });
    res.send({
        data: documentMetadataItems.rows,
        count: documentMetadataItems.count
    });
}));
patientDocumentMetadataRoutes.post('/:id/documentMetadata', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params } = req;
    // TODO: Figure out permissions with Attachment and DocumentMetadata.
    // Presumably, they should be the same as they depend on each other.
    // After it has been figured out, modify the POST /documentMetadata route
    // inside encounter.js and also /createPatientLetter.
    req.checkPermission('write', 'DocumentMetadata');
    // Make sure the specified patient exists
    const patient = await models.Patient.findByPk(params.id);
    if (!patient) {
        throw new _errors.NotFoundError();
    }
    // Create file on the central server
    const { attachmentId, type, metadata } = await (0, _uploadAttachment.uploadAttachment)(req, _constants.DOCUMENT_SIZE_LIMIT);
    const documentMetadataObject = await models.DocumentMetadata.create({
        ...metadata,
        attachmentId,
        type,
        patientId: params.id,
        documentUploadedAt: (0, _dateTime.getCurrentDateTimeString)(),
        source: _constants.DOCUMENT_SOURCES.UPLOADED
    });
    res.send(documentMetadataObject);
}));
patientDocumentMetadataRoutes.delete('/:id/documentMetadata/:documentMetadataId', _deleteModel.deleteDocumentMetadata);
patientDocumentMetadataRoutes.post('/:id/createPatientLetter', (0, _createPatientLetter.createPatientLetter)('Patient', 'patientId'));

//# sourceMappingURL=patientDocumentMetadata.js.map