"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "importRows", {
    enumerable: true,
    get: function() {
        return importRows;
    }
});
const _lodash = require("lodash");
const _sequelize = require("sequelize");
const _yup = require("yup");
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _constants = require("@tamanu/constants");
const _importerEndpoint = require("./importerEndpoint");
const _errors = require("../errors");
const _stats = require("../stats");
const _importSchemas = /*#__PURE__*/ _interop_require_wildcard(require("../importSchemas"));
const _validateTableRows = require("./validateTableRows");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function _getRequireWildcardCache(nodeInterop) {
    if (typeof WeakMap !== "function") return null;
    var cacheBabelInterop = new WeakMap();
    var cacheNodeInterop = new WeakMap();
    return (_getRequireWildcardCache = function(nodeInterop) {
        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
    })(nodeInterop);
}
function _interop_require_wildcard(obj, nodeInterop) {
    if (!nodeInterop && obj && obj.__esModule) {
        return obj;
    }
    if (obj === null || typeof obj !== "object" && typeof obj !== "function") {
        return {
            default: obj
        };
    }
    var cache = _getRequireWildcardCache(nodeInterop);
    if (cache && cache.has(obj)) {
        return cache.get(obj);
    }
    var newObj = {
        __proto__: null
    };
    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
    for(var key in obj){
        if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) {
            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
            if (desc && (desc.get || desc.set)) {
                Object.defineProperty(newObj, key, desc);
            } else {
                newObj[key] = obj[key];
            }
        }
    }
    newObj.default = obj;
    if (cache) {
        cache.set(obj, newObj);
    }
    return newObj;
}
function findFieldName(values, fkField) {
    const fkFieldLower = fkField.toLowerCase();
    const fkFieldCamel = (0, _lodash.camelCase)(fkField);
    const fkFieldUcfirst = (0, _lodash.upperFirst)(fkField);
    const fkFieldSplit = (0, _lodash.lowerCase)(fkField);
    const fkFieldSplitUcwords = (0, _lodash.startCase)(fkFieldSplit);
    if (values[fkField]) return fkField;
    if (values[fkFieldLower]) return fkFieldLower;
    if (values[fkFieldCamel]) return fkFieldCamel;
    if (values[fkFieldUcfirst]) return fkFieldUcfirst;
    if (values[fkFieldSplit]) return fkFieldSplit;
    if (values[fkFieldSplitUcwords]) return fkFieldSplitUcwords;
    return null;
}
// Some models require special logic to fetch find the existing record for a given set of values
const existingRecordLoaders = {
    // most models can just do a simple ID lookup
    default: (Model, { id })=>Model.findByPk(id, {
            paranoid: false
        }),
    // User requires the password field to be explicitly scoped in
    User: (User, { id })=>User.scope('withPassword').findByPk(id, {
            paranoid: false
        }),
    PatientAdditionalData: (PAD, { patientId })=>PAD.findByPk(patientId, {
            paranoid: false
        }),
    // PatientFieldValue model has a composite PK that uses patientId & definitionId
    PatientFieldValue: (PFV, { patientId, definitionId })=>PFV.findOne({
            where: {
                patientId,
                definitionId
            }
        }, {
            paranoid: false
        }),
    // TranslatedString model has a composite PK that uses stringId & language
    TranslatedString: (TS, { stringId, language })=>TS.findOne({
            where: {
                stringId,
                language
            }
        }, {
            paranoid: false
        }),
    ReferenceDataRelation: (RDR, { referenceDataId, referenceDataParentId, type })=>RDR.findOne({
            where: {
                referenceDataId,
                referenceDataParentId,
                type
            }
        }, {
            paranoid: false
        }),
    TaskTemplateDesignation: (TTD, { taskTemplateId, designationId })=>TTD.findOne({
            where: {
                taskTemplateId,
                designationId
            }
        }, {
            paranoid: false
        }),
    UserDesignation: (UD, { userId, designationId })=>UD.findOne({
            where: {
                userId,
                designationId
            }
        }, {
            paranoid: false
        })
};
function loadExisting(Model, values) {
    const loader = existingRecordLoaders[Model.name] || existingRecordLoaders.default;
    return loader(Model, values);
}
function extractRecordName(values, dataType) {
    if (dataType === 'scheduledVaccine') return values.label;
    return values.name;
}
async function importRows({ errors, log, models }, { rows, sheetName, stats: previousStats = {}, foreignKeySchemata = {}, skipExisting = false }, validationContext = {}) {
    const stats = {
        ...previousStats
    };
    log.debug('Importing rows to database', {
        count: rows.length
    });
    if (rows.length === 0) {
        log.debug('Nothing to do, skipping');
        return stats;
    }
    log.debug('Building reverse lookup table');
    const lookup = new Map();
    for (const { model, values: { id, type = null, name = null } } of rows){
        if (!id) continue;
        const kind = model === 'ReferenceData' ? type : model;
        lookup.set(`kind.${kind}-id.${id}`, null);
        if (name) lookup.set(`kind.${kind}-name.${name.toLowerCase()}`, id);
    }
    log.debug('Resolving foreign keys', {
        rows: rows.length
    });
    const resolvedRows = [];
    for (const { model, sheetRow, values } of rows){
        try {
            for (const fkSchema of foreignKeySchemata[model] ?? []){
                const fkFieldName = findFieldName(values, fkSchema.field);
                if (fkFieldName) {
                    const fkFieldValue = values[fkFieldName];
                    const fkNameLowerId = `${(0, _lodash.lowerFirst)(fkSchema.field)}Id`;
                    // This will never return a value since a set's has() shallow compares keys and objects will never be equal in this case
                    const hasLocalId = lookup.has(`kind.${fkSchema.field}-id.${fkFieldValue}`);
                    const idByLocalName = lookup.get(`kind.${fkSchema.field}-name.${fkFieldValue.toLowerCase()}`);
                    if (hasLocalId) {
                        delete values[fkFieldName];
                        values[fkNameLowerId] = fkFieldValue;
                    } else if (idByLocalName) {
                        delete values[fkFieldName];
                        values[fkNameLowerId] = idByLocalName;
                    } else {
                        const hasRemoteId = (fkSchema.model === 'ReferenceData' ? await models.ReferenceData.count({
                            where: {
                                type: fkSchema.types,
                                id: fkFieldValue
                            }
                        }) : await models[fkSchema.model].count({
                            where: {
                                id: fkFieldValue
                            }
                        })) > 0;
                        const idByRemoteName = (fkSchema.model === 'ReferenceData' ? await models.ReferenceData.findOne({
                            where: {
                                type: fkSchema.types,
                                name: {
                                    [_sequelize.Op.iLike]: fkFieldValue
                                }
                            }
                        }) : await models[fkSchema.model].findOne({
                            where: {
                                name: {
                                    [_sequelize.Op.iLike]: fkFieldValue
                                }
                            }
                        }))?.id;
                        if (hasRemoteId) {
                            delete values[fkFieldName];
                            values[fkNameLowerId] = fkFieldValue;
                        } else if (idByRemoteName) {
                            delete values[fkFieldName];
                            values[fkNameLowerId] = idByRemoteName;
                        } else {
                            throw new Error(`valid foreign key expected in column ${fkFieldName} (corresponding to ${fkNameLowerId}) but found: ${fkFieldValue}`);
                        }
                    }
                }
            }
            resolvedRows.push({
                model,
                sheetRow,
                values
            });
        } catch (err) {
            (0, _stats.updateStat)(stats, (0, _stats.statkey)(model, sheetName), 'errored');
            errors.push(new _errors.ForeignkeyResolutionError(sheetName, sheetRow, err));
        }
    }
    if (resolvedRows.length === 0) {
        log.debug('Nothing left, skipping');
        return stats;
    }
    log.debug('Validating data', {
        rows: resolvedRows.length
    });
    const validRows = [];
    for (const { model, sheetRow, values } of resolvedRows){
        try {
            let schemaName;
            if (model === 'ReferenceData') {
                const specificSchemaName = `RD${sheetName}`;
                const specificSchemaExists = !!_importSchemas[specificSchemaName];
                if (specificSchemaExists) {
                    schemaName = specificSchemaName;
                } else {
                    schemaName = 'ReferenceData';
                }
            } else if (model === 'SurveyScreenComponent') {
                // The question type is added to the SSC rows in programImporter/screens.js
                const { type } = values;
                const specificSchemaName = `SSC${type}`;
                const specificSchemaExists = !!_importSchemas[specificSchemaName];
                if (_config.default.validateQuestionConfigs.enabled && specificSchemaExists) {
                    schemaName = specificSchemaName;
                } else {
                    schemaName = 'SurveyScreenComponent';
                }
            } else {
                const specificSchemaExists = !!_importSchemas[model];
                if (specificSchemaExists) {
                    schemaName = model;
                } else {
                    schemaName = 'Base';
                }
            }
            const schema = _importSchemas[schemaName];
            validRows.push({
                model,
                sheetRow,
                values: await schema.validate(values, {
                    abortEarly: false,
                    context: validationContext
                })
            });
        } catch (err) {
            (0, _stats.updateStat)(stats, (0, _stats.statkey)(model, sheetName), 'errored');
            if (err instanceof _yup.ValidationError) {
                for (const valerr of err.errors){
                    errors.push(new _errors.ValidationError(sheetName, sheetRow, valerr));
                }
            }
        }
    }
    if (validRows.length === 0) {
        log.debug('Nothing left, skipping');
        return stats;
    }
    // Check values across the whole spreadsheet
    const pushErrorFn = (model, sheetRow, message)=>{
        (0, _stats.updateStat)(stats, (0, _stats.statkey)(model, sheetName), 'errored');
        errors.push(new _errors.ValidationError(sheetName, sheetRow, message));
    };
    await (0, _validateTableRows.validateTableRows)(models, validRows, pushErrorFn);
    log.debug('Upserting database rows', {
        rows: validRows.length
    });
    const translationData = [];
    for (const { model, sheetRow, values } of validRows){
        const Model = models[model];
        const existing = await loadExisting(Model, values);
        if (existing && skipExisting) {
            (0, _stats.updateStat)(stats, (0, _stats.statkey)(model, sheetName), 'skipped');
            continue;
        }
        try {
            if (existing) {
                await existing.update(values);
                if (values.deletedAt) {
                    if (![
                        'Permission',
                        'SurveyScreenComponent',
                        'UserFacility'
                    ].includes(model)) {
                        throw new _errors.ValidationError(`Deleting ${model} via the importer is not supported`);
                    }
                    await existing.destroy();
                    (0, _stats.updateStat)(stats, (0, _stats.statkey)(model, sheetName), 'deleted');
                } else {
                    if (existing.deletedAt) {
                        await existing.restore();
                        (0, _stats.updateStat)(stats, (0, _stats.statkey)(model, sheetName), 'restored');
                    }
                    (0, _stats.updateStat)(stats, (0, _stats.statkey)(model, sheetName), 'updated');
                }
            } else {
                await Model.create(values);
                (0, _stats.updateStat)(stats, (0, _stats.statkey)(model, sheetName), 'created');
            }
            const dataType = (0, _importerEndpoint.normaliseSheetName)(sheetName, model);
            const isValidTable = model === 'ReferenceData' || (0, _lodash.camelCase)(model) === dataType; // All records in the reference data table are translatable // This prevents join tables from being translated - unsure about this
            const isTranslatable = _constants.TRANSLATABLE_REFERENCE_TYPES.includes(dataType);
            if (isTranslatable && isValidTable) {
                translationData.push([
                    `${_constants.REFERENCE_DATA_TRANSLATION_PREFIX}.${dataType}.${values.id}`,
                    extractRecordName(values, dataType) ?? '',
                    _constants.DEFAULT_LANGUAGE_CODE
                ]);
            }
        } catch (err) {
            (0, _stats.updateStat)(stats, (0, _stats.statkey)(model, sheetName), 'errored');
            errors.push(new _errors.UpsertionError(sheetName, sheetRow, err));
        }
    }
    // Bulk upsert translation defaults
    if (translationData.length > 0) {
        await models.TranslatedString.sequelize.query(`
        INSERT INTO translated_strings (string_id, text, language)
        VALUES ${translationData.map(()=>'(?)').join(',')}
          ON CONFLICT (string_id, language) DO UPDATE SET text = excluded.text;
      `, {
            replacements: translationData,
            type: models.TranslatedString.sequelize.QueryTypes.INSERT
        });
    }
    log.debug('Done with these rows');
    return stats;
}

//# sourceMappingURL=importRows.js.map