"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "LabRequest", {
    enumerable: true,
    get: function() {
        return LabRequest;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _errors = require("@tamanu/shared/errors");
const _Model = require("./Model");
const _buildEncounterLinkedSyncFilter = require("../sync/buildEncounterLinkedSyncFilter");
const _model = require("../types/model");
const _dateTime = require("@tamanu/utils/dateTime");
const _generateDisplayId = require("@tamanu/utils/generateDisplayId");
const _buildSyncLookupSelect = require("../sync/buildSyncLookupSelect");
let LabRequest = class LabRequest extends _Model.Model {
    static initModel({ primaryKey, ...options }, models) {
        super.init({
            id: primaryKey,
            sampleTime: (0, _model.dateTimeType)('sampleTime', {
                allowNull: true
            }),
            requestedDate: (0, _model.dateTimeType)('requestedDate', {
                allowNull: false,
                defaultValue: _dateTime.getCurrentDateTimeString
            }),
            specimenAttached: {
                type: _sequelize.DataTypes.BOOLEAN,
                defaultValue: false
            },
            urgent: {
                type: _sequelize.DataTypes.BOOLEAN,
                defaultValue: false
            },
            status: {
                type: _sequelize.DataTypes.STRING,
                defaultValue: _constants.LAB_REQUEST_STATUSES.RECEPTION_PENDING
            },
            reasonForCancellation: {
                type: _sequelize.DataTypes.STRING
            },
            senaiteId: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            },
            sampleId: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            },
            displayId: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false,
                defaultValue () {
                    return (0, _generateDisplayId.generateDisplayId)();
                }
            },
            publishedDate: (0, _model.dateTimeType)('publishedDate', {
                allowNull: true
            })
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL,
            hooks: {
                afterUpdate: async (labRequest)=>{
                    const shouldPushNotification = [
                        _constants.LAB_REQUEST_STATUSES.INTERIM_RESULTS,
                        _constants.LAB_REQUEST_STATUSES.PUBLISHED,
                        _constants.LAB_REQUEST_STATUSES.INVALIDATED
                    ].includes(labRequest.status);
                    if (shouldPushNotification && labRequest.status !== labRequest.previous('status')) {
                        await models.Notification.pushNotification(_constants.NOTIFICATION_TYPES.LAB_REQUEST, labRequest.dataValues);
                    }
                    const shouldDeleteNotification = [
                        _constants.LAB_REQUEST_STATUSES.DELETED,
                        _constants.LAB_REQUEST_STATUSES.ENTERED_IN_ERROR
                    ].includes(labRequest.status);
                    if (shouldDeleteNotification && labRequest.status !== labRequest.previous('status')) {
                        await models.Notification.destroy({
                            where: {
                                metadata: {
                                    id: labRequest.id
                                }
                            }
                        });
                    }
                }
            }
        });
    }
    static createWithTests(data) {
        return this.sequelize.transaction(async ()=>{
            const { labTestTypeIds = [] } = data;
            if (!labTestTypeIds.length) {
                throw new _errors.InvalidOperationError('A request must have at least one test');
            }
            const { LabTest, LabTestPanelRequest, LabRequestLog } = this.sequelize.models;
            const { labTest, labTestPanelId, userId, ...requestData } = data;
            let newLabRequest;
            if (labTestPanelId) {
                const { id: labTestPanelRequestId } = await LabTestPanelRequest.create({
                    encounterId: data.encounterId,
                    labTestPanelId
                });
                newLabRequest = await this.create({
                    ...requestData,
                    labTestPanelRequestId
                });
            } else {
                newLabRequest = await this.create(requestData);
            }
            await LabRequestLog.create({
                status: newLabRequest.status,
                labRequestId: newLabRequest.id,
                updatedById: userId
            });
            // then create tests
            await Promise.all(labTestTypeIds.map((t)=>LabTest.create({
                    labTestTypeId: t,
                    labRequestId: newLabRequest.id,
                    date: labTest?.date
                })));
            return newLabRequest;
        });
    }
    static initRelations(models) {
        this.belongsTo(models.Department, {
            foreignKey: 'departmentId',
            as: 'department'
        });
        this.belongsTo(models.User, {
            foreignKey: 'collectedById',
            as: 'collectedBy'
        });
        this.belongsTo(models.User, {
            foreignKey: 'requestedById',
            as: 'requestedBy'
        });
        this.belongsTo(models.Encounter, {
            foreignKey: 'encounterId',
            as: 'encounter'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'labTestCategoryId',
            as: 'category'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'labSampleSiteId',
            as: 'site'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'labTestPriorityId',
            as: 'priority'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'labTestLaboratoryId',
            as: 'laboratory'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'specimenTypeId',
            as: 'specimenType'
        });
        this.belongsTo(models.LabTestPanelRequest, {
            foreignKey: 'labTestPanelRequestId',
            as: 'labTestPanelRequest'
        });
        this.hasMany(models.LabTest, {
            foreignKey: 'labRequestId',
            as: 'tests'
        });
        this.hasMany(models.CertificateNotification, {
            foreignKey: 'labRequestId',
            as: 'certificate_notification'
        });
        this.hasMany(models.LabRequestAttachment, {
            foreignKey: 'labRequestId',
            as: 'labRequestAttachments'
        });
        this.hasMany(models.Note, {
            foreignKey: 'recordId',
            as: 'notes',
            constraints: false,
            scope: {
                recordType: this.name
            }
        });
    }
    static getListReferenceAssociations() {
        return [
            'department',
            'requestedBy',
            'category',
            'priority',
            'laboratory',
            'site',
            'collectedBy',
            'specimenType',
            {
                association: 'labTestPanelRequest',
                include: [
                    'labTestPanel'
                ]
            },
            {
                association: 'tests',
                include: [
                    'labTestType'
                ]
            }
        ];
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable, sessionConfig) {
        if (sessionConfig.syncAllLabRequests) {
            return ''; // include all lab requests
        }
        if (patientCount === 0) {
            return null;
        }
        return (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilter)([
            this.tableName,
            'encounters'
        ], markedForSyncPatientsTable);
    }
    static buildSyncLookupQueryDetails() {
        return {
            select: (0, _buildSyncLookupSelect.buildSyncLookupSelect)(this, {
                patientId: 'encounters.patient_id',
                encounterId: `${this.tableName}.encounter_id`,
                isLabRequestValue: 'TRUE'
            }),
            joins: (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilterJoins)([
                this.tableName,
                'encounters'
            ])
        };
    }
    getTests() {
        return this.sequelize.models.LabTest.findAll({
            where: {
                labRequestId: this.id
            }
        });
    }
    getLatestAttachment() {
        return this.sequelize.models.LabRequestAttachment.findOne({
            where: {
                labRequestId: this.id,
                replacedById: null
            }
        });
    }
};

//# sourceMappingURL=LabRequest.js.map