"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "validateProgramDataElementRecords", {
    enumerable: true,
    get: function() {
        return validateProgramDataElementRecords;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _stats = require("../stats");
const _constants = require("@tamanu/constants");
const _visualisationConfigValidation = require("./visualisationConfigValidation");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
// Checks complex chart core question config individually
function validateComplexChartCoreQuestion(programDataElementRecord, surveyScreenComponentRecord) {
    const { id, type } = programDataElementRecord.values;
    const { visibilityStatus } = surveyScreenComponentRecord.values;
    if (_constants.COMPLEX_CORE_DATA_ELEMENT_TYPES.includes(type) === false) {
        throw new Error('Invalid question type');
    }
    if (_constants.CHARTING_CORE_TYPE_TO_ID[type] !== id) {
        throw new Error('Invalid ID for question type');
    }
    const mandatoryTypes = [
        _constants.PROGRAM_DATA_ELEMENT_TYPES.COMPLEX_CHART_INSTANCE_NAME,
        _constants.PROGRAM_DATA_ELEMENT_TYPES.COMPLEX_CHART_DATE
    ];
    const isMandatory = mandatoryTypes.includes(type);
    if (visibilityStatus !== _constants.VISIBILITY_STATUSES.CURRENT && isMandatory) {
        throw new Error(`${type} cannot be hidden`);
    }
}
// Checks that complex chart core questions: are exactly 4 and ordered specifically
function validateComplexChartCore(programDataElementRecords, sheetName, stats, errors) {
    const hasFourQuestions = programDataElementRecords.length === _constants.COMPLEX_CORE_DATA_ELEMENT_TYPES.length;
    const hasCorrectOrder = programDataElementRecords.every((element, index)=>element?.values?.type === _constants.COMPLEX_CORE_DATA_ELEMENT_TYPES[index]);
    if (!hasFourQuestions || !hasCorrectOrder) {
        const error = new Error('Invalid complex chart core questions');
        (0, _stats.updateStat)(stats, (0, _stats.statkey)('ProgramDataElement', sheetName), 'errored', 1);
        errors.push(error);
    }
}
function validateChartingFirstQuestion(programDataElementRecords, sheetName, stats, errors) {
    const { id, code, type } = programDataElementRecords[0].values;
    const hasWrongId = id !== _constants.CHARTING_DATA_ELEMENT_IDS.dateRecorded;
    const hasWrongType = type !== _constants.PROGRAM_DATA_ELEMENT_TYPES.DATE_TIME;
    if (hasWrongId) {
        const error = new Error(`sheetName: ${sheetName}, code: '${code}', First question should have '${_constants.CHARTING_DATA_ELEMENT_IDS.dateRecorded}' as ID`);
        (0, _stats.updateStat)(stats, (0, _stats.statkey)('ProgramDataElement', sheetName), 'errored', 1);
        errors.push(error);
    }
    if (hasWrongType) {
        const error = new Error(`sheetName: ${sheetName}, code: '${code}', First question should be DateTime type`);
        (0, _stats.updateStat)(stats, (0, _stats.statkey)('ProgramDataElement', sheetName), 'errored', 1);
        errors.push(error);
    }
}
function validateProgramDataElementRecords(records, { context, sheetName, stats: previousStats = {}, surveyType }) {
    const { errors } = context;
    const stats = {
        ...previousStats
    };
    const programDataElementRecords = records.filter(({ model })=>model === 'ProgramDataElement');
    if ([
        _constants.SURVEY_TYPES.SIMPLE_CHART,
        _constants.SURVEY_TYPES.COMPLEX_CHART
    ].includes(surveyType)) {
        validateChartingFirstQuestion(programDataElementRecords, sheetName, stats, errors);
    }
    if (surveyType === _constants.SURVEY_TYPES.COMPLEX_CHART_CORE) {
        validateComplexChartCore(programDataElementRecords, sheetName, stats, errors);
    }
    for (const programDataElementRecord of programDataElementRecords){
        const newErrors = [];
        const { values } = programDataElementRecord;
        const { visualisationConfig = '', code: dataElementCode } = values;
        const surveyScreenComponentRecord = records.find((r)=>r.values.dataElementId === values.id) || {};
        const { validationCriteria = '' } = surveyScreenComponentRecord.values;
        try {
            if (_config.default.validateQuestionConfigs.enabled) {
                (0, _visualisationConfigValidation.validateVisualisationConfig)(visualisationConfig, validationCriteria);
            }
            if (surveyType === _constants.SURVEY_TYPES.COMPLEX_CHART_CORE) {
                validateComplexChartCoreQuestion(programDataElementRecord, surveyScreenComponentRecord);
            }
        } catch (e) {
            const error = new Error(`sheetName: ${sheetName}, code: '${dataElementCode}', ${e.message}`);
            newErrors.push(error);
        }
        if (newErrors.length > 0) {
            (0, _stats.updateStat)(stats, (0, _stats.statkey)('ProgramDataElement', sheetName), 'errored', newErrors.length);
            errors.push(...newErrors);
        }
    }
    return stats;
}

//# sourceMappingURL=programDataElementValidation.js.map