"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    TestCSCA: function() {
        return TestCSCA;
    },
    fakeABtoRealAB: function() {
        return fakeABtoRealAB;
    },
    loadCertificateIntoSigner: function() {
        return loadCertificateIntoSigner;
    },
    newKeypairAndCsr: function() {
        return newKeypairAndCsr;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _crypto = /*#__PURE__*/ _interop_require_default(require("crypto"));
const _datefns = require("date-fns");
const _webcrypto = require("@peculiar/webcrypto");
const _asn1js = require("asn1js");
const _pkijs = require("pkijs");
const _constants = require("@tamanu/constants");
const _encodings = require("@tamanu/utils/encodings");
const _localisation = require("../../localisation");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const webcrypto = new _webcrypto.Crypto();
(0, _pkijs.setEngine)('webcrypto', webcrypto, new _pkijs.CryptoEngine({
    name: 'webcrypto',
    crypto: webcrypto,
    subtle: webcrypto.subtle
}));
async function newKeypairAndCsr() {
    const { publicKey, privateKey } = await webcrypto.subtle.generateKey({
        name: 'ECDSA',
        namedCurve: 'P-256'
    }, true, [
        'sign',
        'verify'
    ]);
    const { keySecret, commonName, provider } = _config.default.integrations.signer;
    const countryCode = (await (0, _localisation.getLocalisation)()).country['alpha-2'];
    const csr = new _pkijs.CertificationRequest();
    csr.version = 0;
    csr.attributes = [];
    csr.subject.typesAndValues.push(new _pkijs.AttributeTypeAndValue({
        type: _constants.X502_OIDS.COUNTRY_NAME,
        value: new _asn1js.PrintableString({
            value: countryCode
        })
    }));
    csr.subject.typesAndValues.push(new _pkijs.AttributeTypeAndValue({
        type: _constants.X502_OIDS.COMMON_NAME,
        value: new _asn1js.PrintableString({
            value: commonName
        })
    }));
    if (provider) {
        csr.subject.typesAndValues.push(new _pkijs.AttributeTypeAndValue({
            type: _constants.X502_OIDS.ORGANISATION_NAME,
            value: new _asn1js.PrintableString({
                value: provider
            })
        }));
    }
    // 9303-12 §7.1.3 specifies that the Signer certificate must have certain
    // extensions. However, while CSRs may have extensions, these are NOT
    // transferred to the certificate upon signing. Instead, the intent is that
    // extensions are set by the issuer (the CSCA). Thus, we don't need to add
    // them here.
    await csr.subjectPublicKeyInfo.importKey(publicKey);
    await csr.sign(privateKey, 'SHA-256');
    const packedCsr = Buffer.from(await csr.toSchema().toBER(false));
    const passphrase = Buffer.from(keySecret, 'base64');
    const privateNodeKey = _crypto.default.createPrivateKey({
        key: new Uint8Array(await webcrypto.subtle.exportKey('pkcs8', privateKey)),
        format: 'der',
        type: 'pkcs8'
    });
    return {
        publicKey: fakeABtoRealAB(await webcrypto.subtle.exportKey('spki', publicKey)),
        privateKey: fakeABtoRealAB(privateNodeKey.export({
            type: 'pkcs8',
            format: 'der',
            cipher: 'aes-256-cbc',
            passphrase
        }).buffer),
        request: (0, _encodings.pem)(packedCsr, 'CERTIFICATE REQUEST')
    };
}
function loadCertificateIntoSigner(certificate, workingPeriod = {}) {
    let binCert;
    let txtCert;
    if (typeof certificate === 'string') {
        binCert = (0, _encodings.depem)(certificate, 'CERTIFICATE');
        txtCert = certificate;
    } else if (Buffer.isBuffer(certificate)) {
        binCert = certificate;
        txtCert = (0, _encodings.pem)(certificate, 'CERTIFICATE');
    } else {
        throw new Error('Certificate must be a string (PEM) or Buffer (DER).');
    }
    const asn = (0, _asn1js.fromBER)(fakeABtoRealAB(binCert));
    if (asn.result.error !== '') throw new Error(asn.result.error);
    const cert = new _pkijs.Certificate({
        schema: asn.result
    });
    const validityPeriodStart = cert.notBefore.value;
    const validityPeriodEnd = cert.notAfter.value;
    // The certificate doesn't include the PKUP, so we assume it from which
    // integration is enabled. In future it would be good to get this directly
    // from issuance API or some other way.
    const workingPeriodStart = workingPeriod.start ?? validityPeriodStart;
    let workingPeriodEnd = workingPeriod.end ?? validityPeriodEnd;
    if (!workingPeriod.end) {
        if (_config.default.integrations.vdsNc?.enabled) {
            workingPeriodEnd = (0, _datefns.add)(workingPeriodStart, {
                days: 96
            });
        }
        if (_config.default.integrations.euDcc?.enabled) {
            workingPeriodEnd = (0, _datefns.add)(workingPeriodStart, {
                days: 365
            });
        }
    }
    return {
        workingPeriodStart,
        workingPeriodEnd,
        validityPeriodStart,
        validityPeriodEnd,
        certificate: txtCert
    };
}
function fakeABtoRealAB(fake) {
    return Uint8Array.from(new Uint8Array(fake).values()).buffer;
}
let TestCSCA = class TestCSCA {
    static async generate() {
        const { publicKey, privateKey } = await webcrypto.subtle.generateKey({
            name: 'ECDSA',
            namedCurve: 'P-256'
        }, true, [
            'sign',
            'verify'
        ]);
        const workingPeriodStart = new Date();
        const workingPeriodEnd = (0, _datefns.add)(workingPeriodStart, {
            days: 365 * 4 + 1
        });
        const validityPeriodStart = workingPeriodStart;
        const validityPeriodEnd = (0, _datefns.add)(workingPeriodEnd, {
            days: 365 * 11 + 3
        });
        const cert = new _pkijs.Certificate();
        cert.version = 2;
        cert.issuer.typesAndValues.push(new _pkijs.AttributeTypeAndValue({
            type: _constants.X502_OIDS.COUNTRY_NAME,
            value: new _asn1js.PrintableString({
                value: 'UT'
            })
        }));
        cert.issuer.typesAndValues.push(new _pkijs.AttributeTypeAndValue({
            type: _constants.X502_OIDS.COMMON_NAME,
            value: new _asn1js.Utf8String({
                value: 'UT CA'
            })
        }));
        cert.subject.typesAndValues.push(new _pkijs.AttributeTypeAndValue({
            type: _constants.X502_OIDS.COUNTRY_NAME,
            value: new _asn1js.PrintableString({
                value: 'UT'
            })
        }));
        cert.subject.typesAndValues.push(new _pkijs.AttributeTypeAndValue({
            type: _constants.X502_OIDS.COMMON_NAME,
            value: new _asn1js.Utf8String({
                value: 'UT CA'
            })
        }));
        cert.notBefore = new _pkijs.Time({
            value: validityPeriodStart
        });
        cert.notAfter = new _pkijs.Time({
            value: validityPeriodEnd
        });
        cert.serialNumber = new _asn1js.Integer({
            value: 1
        });
        await cert.subjectPublicKeyInfo.importKey(publicKey);
        const fingerprint = await webcrypto.subtle.digest({
            name: 'SHA-1'
        }, cert.subjectPublicKeyInfo.subjectPublicKey.valueBlock.valueHex);
        const keyID = new _asn1js.OctetString({
            valueHex: fingerprint
        });
        const authKeyID = new _pkijs.AuthorityKeyIdentifier({
            authorityCertIssuer: cert.issuer,
            authorityCertSerialNumber: cert.serialNumber,
            keyIdentifier: new _asn1js.OctetString({
                valueHex: fingerprint
            })
        });
        const basicConstraints = new _pkijs.BasicConstraints({
            cA: true,
            pathLenConstraint: 2
        });
        /* eslint-disable no-bitwise */ const bitArray = new ArrayBuffer(1);
        const bitView = new Uint8Array(bitArray);
        bitView[0] |= 0x02; // Key usage "cRLSign" flag
        bitView[0] |= 0x04; // Key usage "keyCertSign" flag
        /* eslint-enable no-bitwise */ const keyUsage = new _asn1js.BitString({
            valueHex: bitArray
        });
        cert.extensions = new _pkijs.Extensions({
            extensions: [
                new _pkijs.Extension({
                    extnID: _constants.X502_OIDS.BASIC_CONSTRAINTS,
                    critical: true,
                    extnValue: basicConstraints.toSchema().toBER(false),
                    parsedValue: basicConstraints
                }),
                new _pkijs.Extension({
                    extnID: _constants.X502_OIDS.KEY_IDENTIFIER,
                    critical: true,
                    extnValue: keyID.toBER(false),
                    parsedValue: keyID
                }),
                new _pkijs.Extension({
                    extnID: _constants.X502_OIDS.AUTHORITY_KEY_IDENTIFIER,
                    critical: true,
                    extnValue: authKeyID.toSchema().toBER(false),
                    parsedValue: authKeyID
                }),
                new _pkijs.Extension({
                    extnID: _constants.X502_OIDS.KEY_USAGE,
                    critical: false,
                    extnValue: keyUsage.toBER(false),
                    parsedValue: keyUsage
                })
            ]
        });
        await cert.sign(privateKey, 'SHA-256');
        return new TestCSCA(privateKey, publicKey, cert);
    }
    async signCSR(request) {
        const asn = (0, _asn1js.fromBER)(fakeABtoRealAB((0, _encodings.depem)(request, 'CERTIFICATE REQUEST')));
        if (asn.result.error !== '') throw new Error(asn.result.error);
        const csr = new _pkijs.CertificationRequest({
            schema: asn.result
        });
        const workingPeriodStart = new Date();
        const workingPeriodEnd = (0, _datefns.add)(workingPeriodStart, {
            days: 96
        });
        const validityPeriodStart = workingPeriodStart;
        const validityPeriodEnd = (0, _datefns.add)(workingPeriodEnd, {
            days: 365 * 10 + 2
        });
        const cert = new _pkijs.Certificate();
        cert.version = 2;
        cert.issuer = this.certificate.issuer;
        cert.subject = csr.subject;
        cert.notBefore = new _pkijs.Time({
            value: validityPeriodStart
        });
        cert.notAfter = new _pkijs.Time({
            value: validityPeriodEnd
        });
        cert.serialNumber = new _asn1js.Integer({
            value: this.serial += 1
        });
        const fingerprint = await webcrypto.subtle.digest({
            name: 'SHA-1'
        }, this.certificate.subjectPublicKeyInfo.subjectPublicKey.valueBlock.valueHex);
        const authKey = new _pkijs.AuthorityKeyIdentifier({
            authorityCertIssuer: this.certificate.issuer,
            authorityCertSerialNumber: this.certificate.serialNumber,
            keyIdentifier: new _asn1js.OctetString({
                valueHex: fingerprint
            })
        });
        const docType = new _asn1js.Sequence({
            value: [
                new _asn1js.Integer({
                    value: 0
                }),
                new _asn1js.Set({
                    value: [
                        new _asn1js.PrintableString({
                            value: _constants.ICAO_DOCUMENT_TYPES.PROOF_OF_TESTING.DOCTYPE
                        }),
                        new _asn1js.PrintableString({
                            value: _constants.ICAO_DOCUMENT_TYPES.PROOF_OF_VACCINATION.DOCTYPE
                        })
                    ]
                })
            ]
        });
        const extKeyUsage = new _pkijs.ExtKeyUsage({
            keyPurposes: [
                _constants.X502_OIDS.EKU_VDS_NC
            ]
        });
        cert.extensions = [
            new _pkijs.Extension({
                extnID: _constants.X502_OIDS.AUTHORITY_KEY_IDENTIFIER,
                critical: true,
                extnValue: authKey.toSchema().toBER(false),
                parsedValue: authKey
            }),
            new _pkijs.Extension({
                extnID: _constants.X502_OIDS.DOCUMENT_TYPE,
                critical: false,
                extnValue: docType.toBER(false),
                parsedValue: docType
            }),
            new _pkijs.Extension({
                extnID: _constants.X502_OIDS.EXTENDED_KEY_USAGE,
                critical: false,
                extnValue: extKeyUsage.toSchema().toBER(false),
                parsedValue: extKeyUsage
            })
        ];
        cert.subjectPublicKeyInfo = csr.subjectPublicKeyInfo;
        await cert.sign(this.privateKey, 'SHA-256');
        const packed = Buffer.from(await cert.toSchema().toBER(false));
        return (0, _encodings.pem)(packed, 'CERTIFICATE');
    }
    constructor(privateKey, publicKey, certificate){
        this.privateKey = privateKey;
        this.publicKey = publicKey;
        this.certificate = certificate;
        this.serial = 1000;
    }
};

//# sourceMappingURL=Crypto.js.map