"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "Setting", {
    enumerable: true,
    get: function() {
        return Setting;
    }
});
const _sequelize = require("sequelize");
const _lodash = require("lodash");
const _cache = require("@tamanu/settings/cache");
const _constants = require("@tamanu/constants");
const _schema = require("@tamanu/settings/schema");
const _Model = require("./Model");
const _buildSyncLookupSelect = require("../sync/buildSyncLookupSelect");
const SETTINGS_SCOPES_VALUES = Object.values(_constants.SETTINGS_SCOPES);
let Setting = class Setting extends _Model.Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            key: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: false
            },
            value: _sequelize.DataTypes.JSONB,
            scope: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: false,
                defaultValue: _constants.SETTINGS_SCOPES.GLOBAL
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.PULL_FROM_CENTRAL,
            hooks: {
                afterSave () {
                    _cache.settingsCache.reset();
                },
                afterBulkCreate () {
                    _cache.settingsCache.reset();
                },
                afterBulkUpdate () {
                    _cache.settingsCache.reset();
                },
                afterBulkDestroy () {
                    _cache.settingsCache.reset();
                }
            },
            indexes: [
                {
                    // settings_alive_key_unique_cnt
                    // overly broad constraint, narrowed by the next two indices
                    unique: true,
                    fields: [
                        'key',
                        'facility_id',
                        'deleted_at'
                    ]
                },
                {
                    // settings_alive_key_unique_with_facility_idx
                    unique: true,
                    fields: [
                        'key',
                        'facility_id'
                    ],
                    where: {
                        deleted_at: null,
                        facility_id: {
                            [_sequelize.Op.ne]: null
                        }
                    }
                },
                {
                    // settings_alive_key_unique_without_facility_idx
                    unique: true,
                    fields: [
                        'key',
                        'scope'
                    ],
                    where: {
                        deleted_at: null,
                        facility_id: null
                    }
                }
            ]
        });
    }
    static initRelations(models) {
        this.belongsTo(models.Facility, {
            foreignKey: 'facilityId',
            as: 'facility'
        });
    }
    /**
   * IMPORTANT: Duplicated from mobile/models/Setting.ts
   * Please update both places when modify
   */ static async get(key = '', facilityId = null, scopeOverride = null) {
        const determineScope = ()=>{
            if (scopeOverride) {
                return scopeOverride;
            }
            if (facilityId) {
                return _constants.SETTINGS_SCOPES.FACILITY;
            }
            return null;
        };
        const scope = determineScope();
        const settings = await Setting.findAll({
            where: {
                ...key ? {
                    key: {
                        [_sequelize.Op.or]: {
                            [_sequelize.Op.eq]: key,
                            [_sequelize.Op.like]: `${key}.%`
                        }
                    }
                } : {},
                ...scope ? {
                    scope
                } : {},
                facilityId: {
                    ...facilityId ? {
                        [_sequelize.Op.eq]: facilityId
                    } : {
                        [_sequelize.Op.is]: null
                    }
                }
            },
            // we want facility keys to come last so they override global keys
            order: [
                [
                    'key',
                    'ASC'
                ],
                [
                    _sequelize.Sequelize.fn('coalesce', _sequelize.Sequelize.col('facility_id'), '###'),
                    'ASC'
                ]
            ]
        });
        const settingsObject = {};
        for (const currentSetting of settings){
            (0, _lodash.set)(settingsObject, currentSetting.key, currentSetting.value);
        }
        if (key === '') {
            return settingsObject;
        }
        // just return the object or value below the requested key
        // e.g. if schedules.outPatientDischarger was requested, the return object will look like
        // {  schedule: '0 11 * * *', batchSize: 1000 }
        // rather than
        // { schedules: { outPatientDischarger: { schedule: '0 11 * * *', batchSize: 1000 } } }
        return (0, _lodash.get)(settingsObject, key);
    }
    static async set(key = '', value, scope = _constants.SETTINGS_SCOPES.GLOBAL, facilityId = null) {
        const records = buildSettingsRecords(key, value, facilityId, scope);
        const schema = (0, _schema.getScopedSchema)(scope);
        const defaultsForScope = (0, _schema.extractDefaults)(schema);
        const existingSettings = await this.findAll({
            where: {
                key: records.map((r)=>r.key),
                scope,
                facilityId
            },
            paranoid: false
        });
        const existingByKey = (0, _lodash.keyBy)(existingSettings, 'key');
        await Promise.all(records.map(async (record)=>{
            const existing = existingByKey[record.key];
            if (existing) {
                if (existing.deletedAt) {
                    await this.restore({
                        where: {
                            id: existing.id
                        }
                    });
                }
                if (!(0, _lodash.isEqual)(existing.value, record.value)) {
                    // only update existing records that have changed
                    await this.update({
                        value: record.value
                    }, {
                        where: {
                            id: existing.id
                        }
                    });
                }
            } else {
                // only create records for values that differ from the defaults
                if (!(0, _lodash.isEqual)(record.value, (0, _lodash.get)(defaultsForScope, record.key))) {
                    await this.create(record);
                }
            }
        }));
        const keyWhere = key ? {
            [_sequelize.Op.or]: {
                [_sequelize.Op.eq]: key,
                [_sequelize.Op.like]: `${key}.%`
            }
        } : {};
        // delete any records that are no longer needed
        await this.update({
            deletedAt: _sequelize.Sequelize.fn('now')
        }, {
            where: {
                key: {
                    [_sequelize.Op.and]: {
                        ...keyWhere,
                        [_sequelize.Op.notIn]: records.map((r)=>r.key)
                    }
                },
                scope,
                facilityId
            }
        });
    }
    static buildSyncFilter() {
        return `WHERE (facility_id in (:facilityIds) OR scope = '${_constants.SETTINGS_SCOPES.GLOBAL}') AND ${this.tableName}.updated_at_sync_tick > :since`;
    }
    static buildSyncLookupQueryDetails() {
        return {
            select: (0, _buildSyncLookupSelect.buildSyncLookupSelect)(this, {
                facilityId: 'settings.facility_id'
            })
        };
    }
};
function buildSettingsRecords(keyPrefix, value, facilityId, scope = _constants.SETTINGS_SCOPES.GLOBAL) {
    if ((0, _lodash.isPlainObject)(value)) {
        return Object.entries(value).flatMap(([k, v])=>buildSettingsRecords([
                keyPrefix,
                k
            ].filter(Boolean).join('.'), v, facilityId, scope));
    }
    return [
        {
            key: keyPrefix,
            value,
            facilityId,
            scope
        }
    ];
}

//# sourceMappingURL=Setting.js.map