"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "baseDataGenerator", {
    enumerable: true,
    get: function() {
        return baseDataGenerator;
    }
});
const _datefns = require("date-fns");
const _lodash = require("lodash");
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _dateTime = require("@tamanu/utils/dateTime");
const _utilities = require("../utilities");
const _transformAnswers = require("../utilities/transformAnswers");
const _sleepAsync = require("@tamanu/utils/sleepAsync");
const WILLIAM_HOROTO_IDS = [
    'f4a0e3f0-54da-4fc9-a73e-1b72c9ca92a5',
    '4d719b6f-af55-42ac-99b3-5a27cadaab2b',
    '2d574680-e0fc-4956-a37e-121ccb434995',
    'e8a51fc9-28fd-4284-8d00-9ef87bd8d855',
    'c11229a7-b95c-4416-a3ad-560cd75d8f21',
    'cebdd9a4-2744-4ad2-9919-98dc0b15464c'
];
const YIELD_EVERY_N_LOOPS = 100;
const yieldControl = ()=>(0, _sleepAsync.sleepAsync)(20);
const parametersToLabTestSqlWhere = (parameters)=>{
    const defaultWhereClause = {
        '$labRequest.lab_test_category_id$': [
            'labTestCategory-COVID',
            'labTestCategory-COVIDRAT'
        ],
        '$labRequest.status$': {
            [_sequelize.Op.ne]: _constants.LAB_REQUEST_STATUSES.DELETED
        },
        '$labRequest->encounter->patient.id$': {
            [_sequelize.Op.notIn]: WILLIAM_HOROTO_IDS
        }
    };
    if (!parameters || !Object.keys(parameters).length) {
        return defaultWhereClause;
    }
    const whereClause = Object.entries(parameters).filter(([, val])=>val).reduce((where, [key, value])=>{
        const newWhere = {
            ...where
        };
        switch(key){
            case 'village':
                newWhere['$labRequest->encounter->patient.village_id$'] = value;
                break;
            case 'labTestLaboratory':
                newWhere['$labRequest.lab_test_laboratory_id$'] = value;
                break;
            default:
                break;
        }
        return newWhere;
    }, defaultWhereClause);
    return whereClause;
};
const parametersToSurveyResponseSqlWhere = (parameters, { surveyId })=>{
    const defaultWhereClause = {
        '$surveyResponse.survey_id$': surveyId
    };
    if (!parameters || !Object.keys(parameters).length) {
        return defaultWhereClause;
    }
    const whereClause = Object.entries(parameters).filter(([, val])=>val).reduce((where, [key, value])=>{
        const newWhere = {
            ...where
        };
        switch(key){
            case 'village':
                newWhere['$surveyResponse->encounter->patient.village_id$'] = value;
                break;
            default:
                break;
        }
        return newWhere;
    }, defaultWhereClause);
    return whereClause;
};
const getLabTests = async (models, parameters)=>models.LabTest.findAll({
        include: [
            {
                model: models.LabRequest,
                as: 'labRequest',
                where: {
                    status: {
                        [_sequelize.Op.notIn]: [
                            _constants.LAB_REQUEST_STATUSES.DELETED,
                            _constants.LAB_REQUEST_STATUSES.ENTERED_IN_ERROR,
                            _constants.LAB_REQUEST_STATUSES.CANCELLED,
                            _constants.LAB_REQUEST_STATUSES.INVALIDATED
                        ]
                    }
                },
                include: [
                    {
                        model: models.Encounter,
                        as: 'encounter',
                        include: [
                            {
                                model: models.Patient,
                                as: 'patient',
                                include: [
                                    {
                                        model: models.PatientAdditionalData,
                                        as: 'additionalData',
                                        include: [
                                            'ethnicity',
                                            'nationality'
                                        ]
                                    },
                                    'village'
                                ]
                            },
                            {
                                model: models.Location,
                                as: 'location',
                                include: [
                                    'facility'
                                ]
                            }
                        ]
                    },
                    {
                        model: models.ReferenceData,
                        as: 'category'
                    },
                    {
                        model: models.ReferenceData,
                        as: 'priority'
                    },
                    {
                        model: models.ReferenceData,
                        as: 'laboratory'
                    },
                    {
                        model: models.User,
                        as: 'requestedBy'
                    }
                ]
            },
            {
                model: models.ReferenceData,
                as: 'labTestMethod'
            },
            {
                model: models.LabTestType,
                as: 'labTestType'
            }
        ],
        where: parametersToLabTestSqlWhere(parameters),
        order: [
            // The date column only has daily resolution, so will return in non-deterministic order
            [
                'date',
                'ASC'
            ],
            [
                'created_at',
                'ASC'
            ]
        ],
        raw: true,
        nest: true
    });
const getFijiCovidAnswers = async (models, parameters, { surveyId, dateFormat })=>{
    // Use the latest survey responses per patient above to get the corresponding answers
    const answers = await models.SurveyResponseAnswer.findAll({
        where: parametersToSurveyResponseSqlWhere(parameters, {
            surveyId
        }),
        include: [
            {
                model: models.SurveyResponse,
                as: 'surveyResponse',
                include: [
                    {
                        model: models.Encounter,
                        as: 'encounter',
                        // patient is only necessary if a village is specified
                        ...parameters.village ? {
                            include: [
                                {
                                    model: models.Patient,
                                    as: 'patient'
                                }
                            ]
                        } : {}
                    }
                ],
                order: [
                    [
                        'end_time',
                        'ASC'
                    ]
                ]
            }
        ],
        raw: true,
        nest: true
    });
    const components = await models.SurveyScreenComponent.getComponentsForSurvey(surveyId, {
        includeAllVitals: true
    });
    const transformedAnswers = await (0, _transformAnswers.transformAnswers)(models, answers, components, {
        dateFormat
    });
    return transformedAnswers;
};
// Find latest survey response within date range using the answers.
const getLatestPatientAnswerInDateRange = (transformedAnswersByPatientAndDataElement, currentlabTestDate, nextLabTestDate, patientId, dataElementId)=>{
    const patientTransformedAnswers = transformedAnswersByPatientAndDataElement[`${patientId}|${dataElementId}`];
    if (!patientTransformedAnswers) {
        return undefined;
    }
    const sortedLatestToOldestAnswers = patientTransformedAnswers.sort((a1, a2)=>(0, _dateTime.differenceInMilliseconds)((0, _datefns.parseISO)(a2.responseEndTime), (0, _datefns.parseISO)(a1.responseEndTime)));
    const latestAnswer = sortedLatestToOldestAnswers.find((a)=>(0, _datefns.isWithinInterval)((0, _datefns.parseISO)(a.responseEndTime), {
            start: currentlabTestDate,
            end: nextLabTestDate
        }));
    return latestAnswer?.body;
};
const getLabTestRecords = async (labTests, transformedAnswers, parameters, { surveyQuestionCodes, dateFormat, dateFilterBy })=>{
    const transformedAnswersByPatientAndDataElement = (0, _lodash.groupBy)(transformedAnswers, (a)=>`${a.patientId}|${a.dataElementId}`);
    // Group the lab tests by patient so that we can determine the correct sample collection form for each lab request
    // For example, If a patient have 3 lab requests (1st on July 1st, 2nd on July 10th and 3rd on July 15th).
    // For the first request, it should pick the latest sample collection form from July 1st - 9th
    // For the second request, it should pick the latest sample collection forms from July 10th - 14th
    // For the third request, it should pick the latest sample collection form from July 15th onwards.
    const labTestsByPatientId = (0, _lodash.groupBy)(labTests, (labTest)=>labTest?.labRequest?.encounter?.patientId);
    const results = [];
    let totalLoops = 0;
    for (const [patientId, patientLabTests] of Object.entries(labTestsByPatientId)){
        // lab tests were already sorted by 'date' ASC in the sql.
        for(let i = 0; i < patientLabTests.length; i++){
            totalLoops++;
            if (totalLoops % YIELD_EVERY_N_LOOPS === 0) {
                await yieldControl();
            }
            const labTest = patientLabTests[i];
            const currentLabTestDate = (0, _datefns.startOfDay)((0, _datefns.parseISO)(labTest.date));
            const dateToFilterBy = dateFilterBy === 'labRequest.sampleTime' ? (0, _datefns.startOfDay)((0, _datefns.parseISO)(labTest.labRequest.sampleTime)) : currentLabTestDate;
            // Get all lab tests regardless and filter fromDate and toDate in memory
            // to ensure that we have the date range from current lab test to the next lab test correctly.
            if ((0, _datefns.isBefore)(dateToFilterBy, (0, _datefns.startOfDay)(parameters.fromDate ? (0, _datefns.parseISO)(parameters.fromDate) : (0, _datefns.subDays)(new Date(), 30)))) {
                continue;
            }
            if (parameters.toDate && (0, _datefns.isAfter)(dateToFilterBy, (0, _datefns.endOfDay)((0, _datefns.parseISO)(parameters.toDate)))) {
                continue;
            }
            const nextLabTest = patientLabTests[i + 1];
            let nextLabTestDate;
            if (nextLabTest) {
                const nextLabTestTimestamp = (0, _datefns.parseISO)(nextLabTest.date);
                // if next lab test not on the same date (next one on a different date,
                // startOf('day') to exclude the next date when comparing range later
                if (!(0, _datefns.isSameDay)(currentLabTestDate, nextLabTestTimestamp)) {
                    nextLabTestDate = (0, _datefns.startOfDay)(nextLabTestTimestamp);
                } else {
                    // if next lab test on the same date, just use its raw timestamp
                    nextLabTestDate = nextLabTestTimestamp;
                }
            } else {
                // use current time if there's no next lab test
                nextLabTestDate = new Date();
            }
            const { labRequest } = labTest;
            const encounter = labRequest?.encounter;
            const patient = encounter?.patient;
            const village = patient?.village?.name;
            const patientAdditionalData = patient?.additionalData?.[0];
            const formatDate = (date)=>date ? (0, _dateTime.format)(date, dateFormat) : '';
            const labTestRecord = {
                firstName: patient?.firstName,
                lastName: patient?.lastName,
                dob: formatDate(patient?.dateOfBirth),
                sex: patient?.sex,
                patientId: patient?.displayId,
                village,
                labRequestId: labRequest?.displayId,
                labRequestType: labRequest?.category?.name,
                labTestType: labTest?.labTestType?.name,
                status: _constants.LAB_REQUEST_STATUS_CONFIG[labRequest?.status]?.label || labRequest?.status,
                result: labTest.result,
                requestedBy: labRequest?.requestedBy?.displayName,
                submittedDate: formatDate(labTest.date),
                requestedDate: formatDate(labRequest.requestedDate),
                testingDate: formatDate(labTest.completedDate),
                testingTime: labTest.completedDate ? (0, _dateTime.format)((0, _datefns.parseISO)(labTest.completedDate), 'h:mm:ss aa') : '',
                priority: labRequest?.priority?.name,
                testingLaboratory: labRequest?.laboratory?.name,
                laboratoryOfficer: labTest?.laboratoryOfficer,
                labTestMethod: labTest?.labTestMethod?.name,
                additionalDataEthnicity: patientAdditionalData?.ethnicity?.name,
                additionalDataNationality: patientAdditionalData?.nationality?.name,
                additionalDataPassportNumber: patientAdditionalData?.passport,
                sampleTime: labRequest?.sampleTime,
                facilityName: encounter?.location?.facility?.name
            };
            Object.entries(surveyQuestionCodes).forEach(([key, dataElement])=>{
                labTestRecord[key] = getLatestPatientAnswerInDateRange(transformedAnswersByPatientAndDataElement, currentLabTestDate, nextLabTestDate, patientId, dataElement);
            });
            results.push(labTestRecord);
        }
    }
    return results;
};
const baseDataGenerator = async ({ models }, parameters = {}, { surveyId, reportColumnTemplate, surveyQuestionCodes, dateFormat = 'yyyy/MM/dd', dateFilterBy = 'date' })=>{
    const labTests = await getLabTests(models, parameters);
    const transformedAnswers = await getFijiCovidAnswers(models, parameters, {
        surveyId,
        dateFormat
    });
    const reportData = await getLabTestRecords(labTests, transformedAnswers, parameters, {
        surveyQuestionCodes,
        dateFormat,
        dateFilterBy
    });
    return (0, _utilities.generateReportFromQueryData)(reportData, reportColumnTemplate);
};

//# sourceMappingURL=covid-swab-lab-test-list.js.map