"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "Appointment", {
    enumerable: true,
    get: function() {
        return Appointment;
    }
});
const _sequelize = require("sequelize");
const _lodash = require("lodash");
const _constants = require("@tamanu/constants");
const _Model = require("./Model");
const _buildSyncLookupSelect = require("../sync/buildSyncLookupSelect");
const _model = require("../types/model");
let Appointment = class Appointment extends _Model.Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            startTime: (0, _model.dateTimeType)('startTime', {
                allowNull: false
            }),
            endTime: (0, _model.dateTimeType)('endTime'),
            status: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false,
                defaultValue: _constants.APPOINTMENT_STATUSES.CONFIRMED
            },
            typeLegacy: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            },
            isHighPriority: {
                type: _sequelize.DataTypes.BOOLEAN,
                allowNull: false,
                defaultValue: false
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL
        });
    }
    static getListReferenceAssociations() {
        return [
            {
                association: 'patient',
                include: [
                    'village'
                ]
            },
            'clinician',
            {
                association: 'location',
                include: [
                    'locationGroup'
                ]
            },
            'locationGroup',
            'appointmentType',
            'bookingType',
            'encounter',
            'schedule'
        ];
    }
    static initRelations(models) {
        this.belongsTo(models.Patient, {
            as: 'patient',
            foreignKey: 'patientId'
        });
        this.belongsTo(models.User, {
            as: 'clinician',
            foreignKey: 'clinicianId'
        });
        this.belongsTo(models.LocationGroup, {
            as: 'locationGroup',
            foreignKey: 'locationGroupId'
        });
        this.belongsTo(models.Location, {
            as: 'location',
            foreignKey: 'locationId'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'bookingTypeId',
            as: 'bookingType'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'appointmentTypeId',
            as: 'appointmentType'
        });
        this.belongsTo(models.Encounter, {
            foreignKey: 'encounterId',
            as: 'encounter'
        });
        this.belongsTo(models.AppointmentSchedule, {
            foreignKey: 'scheduleId',
            as: 'schedule'
        });
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return `
      LEFT JOIN
        location_groups
      ON
        appointments.location_group_id = location_groups.id
      LEFT JOIN
        locations
      ON appointments.location_id = locations.id
      WHERE
        appointments.patient_id IN (SELECT patient_id FROM ${markedForSyncPatientsTable})
      AND
        COALESCE(location_groups.facility_id, locations.facility_id) IN (:facilityIds)
      AND
        appointments.updated_at_sync_tick > :since
    `;
    }
    static buildSyncLookupQueryDetails() {
        return {
            select: (0, _buildSyncLookupSelect.buildSyncLookupSelect)(this, {
                patientId: `${this.tableName}.patient_id`,
                facilityId: 'COALESCE(location_groups.facility_id, locations.facility_id)'
            }),
            joins: `
        LEFT JOIN location_groups ON appointments.location_group_id = location_groups.id
        LEFT JOIN locations ON appointments.location_id = locations.id
      `
        };
    }
    static async createWithSchedule({ settings, appointmentData, scheduleData }) {
        return this.sequelize.transaction(async ()=>{
            const schedule = await this.sequelize.models.AppointmentSchedule.create(scheduleData);
            const appointments = await schedule.generateRepeatingAppointment(settings, appointmentData);
            return {
                firstAppointment: appointments[0],
                schedule
            };
        });
    }
    /** Convert the appointment to a data object that can be used to create a new appointment. */ toCreateData() {
        return (0, _lodash.omit)(this.get({
            plain: true
        }), [
            'id',
            'createdAt',
            'updatedAt'
        ]);
    }
};

//# sourceMappingURL=Appointment.js.map