import { type HasManyGetAssociationsMixin } from 'sequelize';
import { REPEAT_FREQUENCY } from '@tamanu/constants';
import type { ReadSettings } from '@tamanu/settings/reader';
import { Model } from './Model';
import type { Appointment, AppointmentCreateData } from './Appointment';
import type { InitOptions, Models } from '../types/model';
import type { SyncHookSnapshotChanges, SyncSnapshotAttributes } from 'types/sync';
export type AppointmentScheduleCreateData = Omit<AppointmentSchedule, 'id' | 'createdAt' | 'deletedAt' | 'updatedAtSyncTick'>;
export type WeeklySchedule = AppointmentSchedule & {
    frequency: typeof REPEAT_FREQUENCY.WEEKLY;
    daysOfWeek: [string];
};
export type MonthlySchedule = AppointmentSchedule & {
    frequency: typeof REPEAT_FREQUENCY.MONTHLY;
    daysOfWeek: [string];
    nthWeekday: number;
};
/**
 * Schema to follow iCalendar standard for recurring events.
 * @see https://icalendar.org/iCalendar-RFC-5545/3-3-10-recurrence-rule.html
 */
export declare class AppointmentSchedule extends Model {
    id: string;
    untilDate?: string;
    generatedUntilDate?: string;
    cancelledAtDate?: string;
    interval: number;
    frequency: keyof typeof REPEAT_FREQUENCY;
    daysOfWeek?: [string];
    nthWeekday?: number;
    occurrenceCount?: number;
    isFullyGenerated: boolean;
    getAppointments: HasManyGetAssociationsMixin<Appointment>;
    static initModel({ primaryKey, ...options }: InitOptions): void;
    static initRelations(models: Models): void;
    static buildPatientSyncFilter(patientCount: number, markedForSyncPatientsTable: string): string | null;
    static buildSyncLookupQueryDetails(): {
        select: string;
        joins: string;
    };
    static incomingSyncHook(changes: SyncSnapshotAttributes[]): Promise<SyncHookSnapshotChanges | undefined>;
    isDifferentFromSchedule(scheduleData: AppointmentScheduleCreateData): boolean;
    /**
     * End the schedule at the given appointment.
     * This will cancel all appointments starting from the given appointment.
     * The schedule will then be marked as fully generated and the untilDate will be set to the
     * startTime of the latest non-cancelled appointment.
     * @param appointment All appointments starting from this appointment will be cancelled
     */
    endAtAppointment(appointment: Appointment): Promise<this>;
    /**
     * Modify all appointments starting from the given appointment.
     * @param appointment The appointment to start modifying from
     * @param appointmentData The data to update the appointments with
     */
    modifyFromAppointment(appointment: Appointment, appointmentData: AppointmentCreateData): Promise<[affectedCount: number]>;
    /**
     * Generate repeating appointments based on the schedule parameters and the initial appointment data.
     * When the generation is complete, the schedule is marked as fully generated.
     * Otherwise the schedule continues to generate via the scheduled task GenerateRepeatingAppointments
     * @param settings Settings reader
     * @param initialAppointmentData Optional initial appointment data to start the generation
     */
    generateRepeatingAppointment(settings: ReadSettings, initialAppointmentData?: AppointmentCreateData): Promise<Appointment[]>;
}
//# sourceMappingURL=AppointmentSchedule.d.ts.map