"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "Location", {
    enumerable: true,
    get: function() {
        return Location;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _errors = require("@tamanu/shared/errors");
const _Model = require("./Model");
let Location = class Location extends _Model.Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            code: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false,
                unique: true
            },
            name: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false
            },
            visibilityStatus: {
                type: _sequelize.DataTypes.TEXT,
                defaultValue: _constants.VISIBILITY_STATUSES.CURRENT
            },
            maxOccupancy: {
                type: _sequelize.DataTypes.INTEGER,
                allowNull: true,
                validate: {
                    isValidInt (value) {
                        if (value && value !== 1) {
                            // Currently max occupancy above 1 is unimplemented
                            throw new _errors.InvalidOperationError('A location must have a max occupancy of 1 or null for unrestricted occupancy.');
                        }
                    }
                }
            }
        }, {
            ...options,
            validate: {
                mustHaveFacility () {
                    if (!this.deletedAt && !this.facilityId) {
                        throw new _errors.InvalidOperationError('A location must have a facility.');
                    }
                }
            },
            syncDirection: _constants.SYNC_DIRECTIONS.PULL_FROM_CENTRAL,
            indexes: [
                {
                    unique: true,
                    fields: [
                        'code'
                    ]
                }
            ]
        });
    }
    static initRelations(models) {
        this.hasMany(models.Encounter, {
            foreignKey: 'locationId'
        });
        this.hasMany(models.Procedure, {
            foreignKey: 'locationId'
        });
        this.hasMany(models.ImagingRequest, {
            foreignKey: 'locationId'
        });
        this.belongsTo(models.Facility, {
            foreignKey: 'facilityId',
            as: 'facility'
        });
        this.belongsTo(models.LocationGroup, {
            foreignKey: 'locationGroupId',
            as: 'locationGroup'
        });
        this.hasMany(models.Encounter, {
            foreignKey: 'plannedLocationId',
            as: 'plannedMoves'
        });
    }
    static formatFullLocationName({ locationGroup, name }) {
        return locationGroup ? `${locationGroup.name}, ${name}` : name;
    }
    static parseFullLocationName(text) {
        const match = text.match(RegExp("(?<group>[^,]*(?=,\\s))?(,\\s)?(?<location>.*)"));
        const group = match?.groups?.group;
        const location = match?.groups?.location;
        return {
            group,
            location
        };
    }
    async getAvailability() {
        const { Encounter } = this.sequelize.models;
        /**
     * If a locations maxOccupancy is null there is no limit to the number of patients that can be assigned
     * to location, there will be no warnings and the location will always be available.
     */ if (this.maxOccupancy === null) return _constants.LOCATION_AVAILABILITY_STATUS.AVAILABLE;
        const openEncounters = await Encounter.count({
            where: {
                locationId: this.id,
                endDate: null
            }
        });
        if (openEncounters > 0) {
            return _constants.LOCATION_AVAILABILITY_STATUS.OCCUPIED;
        }
        const plannedEncounters = await Encounter.count({
            where: {
                plannedLocationId: this.id,
                endDate: null
            }
        });
        if (plannedEncounters > 0) {
            return _constants.LOCATION_AVAILABILITY_STATUS.RESERVED;
        }
        return _constants.LOCATION_AVAILABILITY_STATUS.AVAILABLE;
    }
    static buildSyncFilter() {
        return null; // syncs everywhere
    }
    static buildSyncLookupQueryDetails() {
        return null; // syncs everywhere
    }
};

//# sourceMappingURL=Location.js.map