"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "getValues", {
    enumerable: true,
    get: function() {
        return getValues;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _constants = require("@tamanu/constants");
const _fhirTypes = require("@tamanu/shared/services/fhirTypes");
const _fhir = require("@tamanu/shared/utils/fhir");
const _utils = require("../utils");
const _extensions = require("../extensions");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
async function getValues(upstream, models) {
    const { Patient } = models;
    if (upstream instanceof Patient) return getValuesFromPatient(upstream, models);
    throw new Error(`Invalid upstream type for patient ${upstream.constructor.name}`);
}
async function getValuesFromPatient(upstream, models) {
    const links = await mergeLinks(upstream, models);
    return {
        extension: extension(upstream),
        identifier: identifiers(upstream),
        active: (0, _utils.activeFromVisibility)(upstream),
        name: names(upstream),
        telecom: telecoms(upstream),
        gender: upstream.sex,
        birthDate: (0, _fhir.formatFhirDate)(upstream.dateOfBirth, _constants.FHIR_DATETIME_PRECISION.DAYS),
        deceasedDateTime: (0, _fhir.formatFhirDate)(upstream.dateOfDeath, _constants.FHIR_DATETIME_PRECISION.DAYS),
        address: addresses(upstream),
        link: links,
        lastUpdated: new Date(),
        resolved: links.every(({ other })=>other.isResolved())
    };
}
// eslint-disable-next-line no-unused-vars
function compactBy(array, access = (value)=>Boolean(value)) {
    return array.filter(access);
}
function extension(patient) {
    return [
        ...(0, _extensions.nzEthnicity)(patient)
    ];
}
function identifiers(patient) {
    const additionalData = patient?.additionalData?.[0];
    return compactBy([
        {
            use: 'usual',
            value: patient.displayId,
            assigner: new _fhirTypes.FhirReference({
                display: _config.default.hl7.assigners.patientDisplayId
            }),
            system: _config.default.hl7.dataDictionaries.patientDisplayId
        },
        {
            use: 'secondary',
            assigner: new _fhirTypes.FhirReference({
                display: _config.default.hl7.assigners.patientPassport
            }),
            value: additionalData?.passportNumber
        },
        {
            use: 'secondary',
            assigner: new _fhirTypes.FhirReference({
                display: _config.default.hl7.assigners.patientDrivingLicense
            }),
            value: additionalData?.drivingLicense
        }
    ], (x)=>!!x.value).map((i)=>new _fhirTypes.FhirIdentifier(i));
}
function names(patient) {
    const additionalData = patient?.additionalData?.[0];
    return compactBy([
        {
            use: 'official',
            prefix: compactBy([
                additionalData?.title
            ]),
            family: patient.lastName,
            given: compactBy([
                patient.firstName,
                patient.middleName
            ])
        },
        patient.culturalName && {
            use: 'nickname',
            text: patient.culturalName
        }
    ]).map((i)=>new _fhirTypes.FhirHumanName(i));
}
function telecoms(patient) {
    const additionalData = patient?.additionalData?.[0];
    return compactBy([
        additionalData?.primaryContactNumber,
        additionalData?.secondaryContactNumber
    ]).map((value, index)=>new _fhirTypes.FhirContactPoint({
            system: 'phone',
            rank: index + 1,
            value
        }));
}
function addresses(patient) {
    const additionalData = patient?.additionalData?.[0];
    const { cityTown, streetVillage } = additionalData || {};
    return [
        new _fhirTypes.FhirAddress({
            type: 'physical',
            use: 'home',
            city: cityTown,
            line: [
                streetVillage
            ]
        })
    ];
}
async function mergeLinks(patient, models) {
    const links = [];
    // Populates "upstream" links, which must be resolved to FHIR resource links
    // after materialisation by calling FhirResource.resolveUpstreams().
    if (patient.mergedIntoId) {
        const mergeTarget = await patient.getUltimateMergedInto();
        if (mergeTarget) {
            links.push(new _fhirTypes.FhirPatientLink({
                type: 'replaced-by',
                other: await _fhirTypes.FhirReference.to(models.FhirPatient, mergeTarget.id, {
                    display: mergeTarget.displayId
                })
            }));
        }
    }
    const down = await patient.getMergedDown();
    for (const mergedPatient of down){
        if (mergedPatient.mergedIntoId === patient.id) {
            // if it's a merge directly into this patient
            links.push(new _fhirTypes.FhirPatientLink({
                type: 'replaces',
                other: await _fhirTypes.FhirReference.to(models.FhirPatient, mergedPatient.id, {
                    display: mergedPatient.displayId
                })
            }));
        } else {
            links.push(new _fhirTypes.FhirPatientLink({
                type: 'seealso',
                other: await _fhirTypes.FhirReference.to(models.FhirPatient, mergedPatient.id, {
                    display: mergedPatient.displayId
                })
            }));
        }
    }
    return links;
}

//# sourceMappingURL=getValues.js.map