import { SYNC_DIRECTIONS } from '@tamanu/constants';
import { DataTypes } from 'sequelize';
import { getCurrentDateTimeString } from '@tamanu/utils/dateTime';
import { Model } from './Model';
import { dateTimeType } from '../types/model';
import { buildSyncLookupSelect } from '../sync';
export class PatientProgramRegistrationCondition extends Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            date: dateTimeType('date', {
                allowNull: false,
                defaultValue: getCurrentDateTimeString
            }),
            deletionDate: dateTimeType('deletionDate', {
                defaultValue: null
            }),
            conditionCategory: {
                type: DataTypes.STRING,
                defaultValue: 'unknown',
                allowNull: false
            },
            reasonForChange: {
                type: DataTypes.STRING,
                allowNull: true
            }
        }, {
            ...options,
            syncDirection: SYNC_DIRECTIONS.BIDIRECTIONAL
        });
    }
    static initRelations(models) {
        this.belongsTo(models.PatientProgramRegistration, {
            foreignKey: {
                name: 'patientProgramRegistrationId',
                allowNull: false
            },
            as: 'patientProgramRegistration'
        });
        this.belongsTo(models.ProgramRegistryCondition, {
            foreignKey: 'programRegistryConditionId',
            as: 'programRegistryCondition'
        });
        this.belongsTo(models.User, {
            foreignKey: 'clinicianId',
            as: 'clinician'
        });
        this.belongsTo(models.User, {
            foreignKey: 'deletionClinicianId',
            as: 'deletionClinician'
        });
    }
    static getFullReferenceAssociations() {
        return [
            'programRegistryCondition'
        ];
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return `
      WHERE patient_program_registration_id IN (
        SELECT id
        FROM patient_program_registrations
        WHERE patient_id IN (SELECT patient_id FROM ${markedForSyncPatientsTable})
      )
      AND ${this.tableName}.updated_at_sync_tick > :since
    `;
    }
    static buildSyncLookupQueryDetails() {
        return {
            select: buildSyncLookupSelect(this, {
                patientId: 'patient_program_registrations.patient_id'
            }),
            joins: [
                `LEFT JOIN patient_program_registrations ON ${this.tableName}.patient_program_registration_id = patient_program_registrations.id`
            ]
        };
    }
}

//# sourceMappingURL=PatientProgramRegistrationCondition.js.map