"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "tasks", {
    enumerable: true,
    get: function() {
        return taskRoutes;
    }
});
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _errors = require("@tamanu/shared/errors");
const _constants = require("@tamanu/constants");
const _zod = require("zod");
const _sequelize = require("sequelize");
const _uuid = require("uuid");
const _dateTime = require("@tamanu/utils/dateTime");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const taskRoutes = _express.default.Router();
/**
 * Mark task as completed
 * Only tasks in TODO & NON_COMPLETED status can be marked as completed
 */ const taskCompletionInputSchema = _zod.z.object({
    taskIds: _zod.z.string().array().min(1),
    completedByUserId: _zod.z.string(),
    completedTime: _dateTime.datetimeCustomValidation,
    completedNote: _zod.z.string().optional()
});
taskRoutes.post('/completed', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'Tasking');
    const { taskIds, ...completedInfo } = await taskCompletionInputSchema.parseAsync(req.body);
    //validate task
    const allowedStatuses = [
        _constants.TASK_STATUSES.TODO,
        _constants.TASK_STATUSES.NON_COMPLETED
    ];
    const tasks = await req.models.Task.findAll({
        where: {
            id: {
                [_sequelize.Op.in]: taskIds
            },
            status: {
                [_sequelize.Op.in]: allowedStatuses
            }
        },
        attributes: [
            'id',
            'status'
        ]
    });
    if (tasks?.length !== taskIds.length) {
        throw new _errors.ForbiddenError(`Some of selected tasks are not in ${allowedStatuses.join(', ')} status`);
    }
    //update task
    await req.models.Task.update({
        ...completedInfo,
        status: _constants.TASK_STATUSES.COMPLETED
    }, {
        where: {
            id: {
                [_sequelize.Op.in]: taskIds
            }
        }
    });
    res.status(204).json();
}));
/**
 * Mark task as not completed
 * Only tasks in TODO & COMPLETED status can be marked as not completed
 */ const taskNonCompletionInputSchema = _zod.z.object({
    taskIds: _zod.z.string().array().min(1),
    notCompletedByUserId: _zod.z.string(),
    notCompletedTime: _dateTime.datetimeCustomValidation,
    notCompletedReasonId: _zod.z.string().optional()
});
taskRoutes.put('/notCompleted', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'Tasking');
    const { taskIds, ...notCompletedInfo } = await taskNonCompletionInputSchema.parseAsync(req.body);
    //validate not completed reason
    if (notCompletedInfo.notCompletedReasonId) {
        const notCompleteReason = await req.models.ReferenceData.findByPk(notCompletedInfo.notCompletedReasonId, {
            where: {
                type: _constants.REFERENCE_TYPES.TASK_NOT_COMPLETED_REASON
            }
        });
        if (!notCompleteReason) throw new _errors.NotFoundError('Not completed reason not found');
    }
    //validate task
    const allowedStatuses = [
        _constants.TASK_STATUSES.TODO,
        _constants.TASK_STATUSES.COMPLETED
    ];
    const tasks = await req.models.Task.findAll({
        where: {
            id: {
                [_sequelize.Op.in]: taskIds
            },
            status: {
                [_sequelize.Op.in]: allowedStatuses
            }
        },
        attributes: [
            'id',
            'status'
        ]
    });
    if (tasks?.length !== taskIds.length) {
        throw new _errors.ForbiddenError(`Some of selected tasks are not in ${allowedStatuses.join(', ')} status`);
    }
    //update tasks
    await req.models.Task.update({
        ...notCompletedInfo,
        status: _constants.TASK_STATUSES.NON_COMPLETED
    }, {
        where: {
            id: {
                [_sequelize.Op.in]: taskIds
            }
        }
    });
    res.status(204).json();
}));
/**
 * Delete a task
 * Only allow to delete task with TODO status
 */ const taskDeletionInputSchema = _zod.z.object({
    taskIds: _zod.z.string().array().min(1),
    deletedByUserId: _zod.z.string(),
    deletedTime: _dateTime.datetimeCustomValidation,
    deletedReasonId: _zod.z.string().optional()
});
taskRoutes.delete('/', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('delete', 'Tasking');
    const { taskIds, ...deletedInfo } = await taskDeletionInputSchema.parseAsync(req.query);
    //validate deleted reason
    if (deletedInfo.deletedReasonId) {
        const deletedReason = await req.models.ReferenceData.findByPk(deletedInfo.deletedReasonId, {
            where: {
                type: _constants.REFERENCE_TYPES.TASK_DELETION_REASON
            }
        });
        if (!deletedReason) throw new _errors.NotFoundError('Deleted reason not found');
    }
    const deletionReasonForFutureTasks = await req.models.ReferenceData.findByPk(_constants.TASK_DELETE_BY_SYSTEM_REASON);
    //validate task
    const allowedStatuses = [
        _constants.TASK_STATUSES.TODO
    ];
    const tasks = await req.models.Task.findAll({
        where: {
            id: {
                [_sequelize.Op.in]: taskIds
            },
            status: {
                [_sequelize.Op.in]: allowedStatuses
            }
        },
        attributes: [
            'id',
            'status',
            'dueTime',
            'frequencyValue',
            'frequencyUnit',
            'parentTaskId'
        ]
    });
    if (tasks?.length !== taskIds.length) throw new _errors.ForbiddenError('Some of selected tasks are not in TODO status');
    await req.db.transaction(async ()=>{
        //update deletion info
        await req.models.Task.update(deletedInfo, {
            where: {
                id: {
                    [_sequelize.Op.in]: taskIds
                }
            }
        });
        //delete tasks
        for (const task of tasks){
            if (task.isRepeatingTask()) {
                const parentTask = !task.parentTaskId ? task : await req.models.Task.findOne({
                    where: {
                        id: task.parentTaskId
                    }
                });
                if (parentTask) {
                    parentTask.endTime = new Date(new Date(task.dueTime).getTime() - 1);
                    parentTask.deletedReasonForSyncId = deletionReasonForFutureTasks?.id ?? null;
                    await parentTask.save();
                    await req.models.Task.update({
                        deletedByUserId: _constants.SYSTEM_USER_UUID,
                        deletedTime: (0, _dateTime.getCurrentDateTimeString)(),
                        deletedReasonId: deletionReasonForFutureTasks?.id ?? null
                    }, {
                        where: {
                            parentTaskId: parentTask.id,
                            id: {
                                [_sequelize.Op.ne]: task.id
                            },
                            dueTime: {
                                [_sequelize.Op.gt]: parentTask.endTime
                            },
                            status: _constants.TASK_STATUSES.TODO
                        }
                    });
                    await req.models.Task.destroy({
                        where: {
                            parentTaskId: parentTask.id,
                            id: {
                                [_sequelize.Op.ne]: task.id
                            },
                            dueTime: {
                                [_sequelize.Op.gt]: parentTask.endTime
                            },
                            status: _constants.TASK_STATUSES.TODO
                        },
                        individualHooks: true
                    });
                }
            }
            await task.destroy();
        }
    });
    res.status(204).json();
}));
/**
 * Mark task as todo
 * Only tasks in COMPLETED & NON_COMPLETED status can be marked as todo
 * - Only allow to set as todo for tasks that are not older than 48 hours
 * - Copy info from the selected task and set the new task status as todo, set todo info to the new task
 * - Delete the selected task
 */ const taskTodoInputSchema = _zod.z.object({
    taskIds: _zod.z.string().array().min(1),
    todoByUserId: _zod.z.string(),
    todoTime: _dateTime.datetimeCustomValidation.refine((datetime)=>new Date().getTime() - new Date(datetime).getTime() < 2 * 86400000, 'Task is older than 48 hours'),
    todoNote: _zod.z.string().optional()
});
taskRoutes.put('/todo', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'Tasking');
    const { taskIds, ...todoInfo } = await taskTodoInputSchema.parseAsync(req.body);
    //validate task
    const tasks = await req.models.Task.findAll({
        where: {
            id: {
                [_sequelize.Op.in]: taskIds
            }
        },
        attributes: [
            'id',
            'name',
            'dueTime',
            'endTime',
            'requestTime',
            'status',
            'note',
            'frequencyValue',
            'frequencyUnit',
            'highPriority',
            'parentTaskId',
            'encounterId',
            'requestedByUserId'
        ],
        include: [
            'designations'
        ]
    });
    if (!tasks?.length) throw new _errors.NotFoundError('No tasks not found');
    const allowedStatuses = [
        _constants.TASK_STATUSES.COMPLETED,
        _constants.TASK_STATUSES.NON_COMPLETED
    ];
    if (!tasks.every((task)=>allowedStatuses.includes(task.status))) throw new _errors.ForbiddenError(`Task is not in ${allowedStatuses.join(', ')} status`);
    const updateParentIdList = [];
    await req.db.transaction(async ()=>{
        for (const task of tasks){
            //delete the selected task
            await task.destroy();
            //copy info from the selected task and set the new task as todo with todo info
            const newId = (0, _uuid.v4)();
            await req.models.Task.create({
                ...task.dataValues,
                id: newId,
                status: _constants.TASK_STATUSES.TODO,
                ...todoInfo,
                deletedAt: null
            });
            await req.models.TaskDesignation.bulkCreate(task.dataValues.designations.map((designation)=>({
                    designationId: designation.id,
                    taskId: newId
                })));
            if (task.isRepeatingTask() && !task.parentTaskId) {
                updateParentIdList.push({
                    newId: newId,
                    oldId: task.id
                });
            }
        }
        for (const { newId, oldId } of updateParentIdList){
            await req.models.Task.update({
                parentTaskId: newId
            }, {
                where: {
                    parentTaskId: oldId
                }
            });
        }
    });
    res.status(204).json();
}));
const tasksCreationSchema = _zod.z.object({
    startTime: _dateTime.datetimeCustomValidation,
    encounterId: _zod.z.string().uuid(),
    requestedByUserId: _zod.z.string(),
    requestTime: _dateTime.datetimeCustomValidation,
    note: _zod.z.string().optional(),
    tasks: _zod.z.object({
        name: _zod.z.string(),
        frequencyValue: _zod.z.number().optional(),
        frequencyUnit: _zod.z.string().optional(),
        durationValue: _zod.z.number().optional(),
        durationUnit: _zod.z.string().optional(),
        designationIds: _zod.z.string().array().optional(),
        highPriority: _zod.z.boolean()
    }).array()
});
taskRoutes.post('/', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('create', 'Tasking');
    const { startTime, requestedByUserId, requestTime, encounterId, note, tasks } = await tasksCreationSchema.parseAsync(req.body);
    const { models, db } = req;
    const { Task, TaskDesignation } = models;
    await db.transaction(async ()=>{
        const tasksData = tasks.map((task)=>{
            const designations = task.designationIds.map((designation)=>({
                    id: designation
                }));
            return {
                ...task,
                id: (0, _uuid.v4)(),
                designations,
                dueTime: startTime,
                requestedByUserId,
                requestTime,
                encounterId,
                note
            };
        });
        const createdTasks = await Task.bulkCreate(tasksData);
        const taskDesignationAssociations = tasksData.flatMap((task)=>{
            return task.designations.map((designation)=>({
                    taskId: task.id,
                    designationId: designation.id
                }));
        });
        await TaskDesignation.bulkCreate(taskDesignationAssociations);
        const hasRepeatedTasks = tasksData.some((task)=>task.frequencyValue && task.frequencyUnit);
        if (hasRepeatedTasks) {
            await Task.generateRepeatingTasks(tasksData);
        }
        res.send(createdTasks);
    });
}));

//# sourceMappingURL=tasks.js.map