"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "callWithBackoff", {
    enumerable: true,
    get: function() {
        return callWithBackoff;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _logging = require("@tamanu/shared/services/logging");
const _sleepAsync = require("@tamanu/utils/sleepAsync");
const _errors = require("@tamanu/shared/errors");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const IRRECOVERABLE_ERRORS = [
    _errors.BadAuthenticationError,
    _errors.FacilityAndSyncVersionIncompatibleError
];
const isErrorOnIrrecoverableList = (e)=>IRRECOVERABLE_ERRORS.some((irrecErr)=>e instanceof irrecErr);
const is4xx = (e)=>e.centralServerResponse?.status >= 400 && e.centralServerResponse?.status < 500;
const isInsufficientStorage = (e)=>e.centralServerResponse?.message === 'InsufficientStorage';
const isSyncSessionFailure = (e)=>e.centralServerResponse?.message.startsWith('Sync session');
const isIrrecoverable = (e)=>{
    return isErrorOnIrrecoverableList(e) || is4xx(e) || isInsufficientStorage(e) || isSyncSessionFailure(e);
};
const callWithBackoff = async (fn, { maxAttempts = _config.default.sync.backoff.maxAttempts, maxWaitMs = _config.default.sync.backoff.maxWaitMs, multiplierMs = _config.default.sync.backoff.multiplierMs } = {})=>{
    if (!Number.isFinite(maxAttempts) || maxAttempts < 1) {
        throw new Error(`callWithBackoff: maxAttempts must be a finite integer, instead got ${maxAttempts}`);
    }
    let lastN = 0;
    let secondLastN = 0;
    let attempt = 0;
    const overallStartMs = Date.now();
    // eslint-disable-next-line no-constant-condition
    while(true){
        attempt += 1;
        const attemptStartMs = Date.now();
        try {
            _logging.log.debug(`callWithBackoff: started`, {
                attempt,
                maxAttempts
            });
            const result = await fn();
            const now = Date.now();
            const attemptMs = now - attemptStartMs;
            const totalMs = now - overallStartMs;
            _logging.log.debug(`callWithBackoff: succeeded`, {
                attempt,
                maxAttempts,
                time: `${attemptMs}ms`,
                totalTime: `${totalMs}ms`
            });
            return result;
        } catch (e) {
            // throw if the error is irrecoverable
            if (isIrrecoverable(e)) {
                _logging.log.error(`callWithBackoff: failed, error was irrecoverable`, {
                    attempt,
                    maxAttempts,
                    stack: e.stack
                });
                throw e;
            }
            // throw if we've exceeded our maximum retries
            if (attempt >= maxAttempts) {
                _logging.log.error(`callWithBackoff: failed, max retries exceeded`, {
                    attempt,
                    maxAttempts,
                    stack: e.stack
                });
                throw e;
            }
            // otherwise, calculate the next backoff delay
            [secondLastN, lastN] = [
                lastN,
                Math.max(lastN + secondLastN, 1)
            ];
            const delay = Math.min(lastN * multiplierMs, maxWaitMs);
            _logging.log.warn(`callWithBackoff: failed, retrying`, {
                attempt,
                maxAttempts,
                retryingIn: `${delay}ms`,
                stack: e.stack
            });
            await (0, _sleepAsync.sleepAsync)(delay);
        }
    }
};

//# sourceMappingURL=callWithBackoff.js.map