// run any post migration checks or side effects
// so far, this is just adding the updated_at_sync_tick column and trigger to all new tables
"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    runPostMigration: function() {
        return runPostMigration;
    },
    runPreMigration: function() {
        return runPreMigration;
    },
    tablesWithoutColumn: function() {
        return tablesWithoutColumn;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _sequelize = require("sequelize");
const _selectFacilityIds = require("@tamanu/utils/selectFacilityIds");
const _constants = require("./constants");
const _constants1 = require("../../sync/constants");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const tableNameMatch = (schema, table, matches)=>{
    const matchTableSchemas = matches.map((match)=>match.split('.')).map(([excludeSchema, excludeTable])=>({
            schema: excludeSchema,
            table: excludeTable
        }));
    const wholeSchemaMatches = matchTableSchemas.filter(({ table: matchTable })=>matchTable === '*').map(({ schema: matchSchema })=>matchSchema);
    if (wholeSchemaMatches.includes(schema)) {
        return true;
    }
    return matchTableSchemas.some(({ schema: matchSchema, table: matchTable })=>schema === matchSchema && table === matchTable);
};
const tablesWithoutColumn = (sequelize, column, excludes = _constants.NON_SYNCING_TABLES)=>{
    return sequelize.query(`
    SELECT
      pg_namespace.nspname as schema,
      pg_class.relname as table
    FROM pg_catalog.pg_class
    JOIN pg_catalog.pg_namespace
      ON pg_class.relnamespace = pg_namespace.oid
    LEFT JOIN pg_catalog.pg_attribute
    ON pg_attribute.attrelid = pg_class.oid
      AND pg_attribute.attname = $column
    WHERE pg_namespace.nspname IN ('public', 'logs')
      AND pg_class.relkind = 'r'
      AND pg_attribute.attname IS NULL
      AND (pg_namespace.nspname || '.' || pg_class.relname) NOT IN ($excludes);
  `, {
        type: _sequelize.QueryTypes.SELECT,
        bind: {
            column,
            excludes: _constants.NON_SYNCING_TABLES
        }
    }).then((rows)=>rows.map((row)=>({
                schema: row.schema,
                table: row.table
            })).filter(({ schema, table })=>!tableNameMatch(schema, table, excludes)));
};
const tablesWithoutTrigger = (sequelize, prefix, suffix, excludes = _constants.NON_SYNCING_TABLES)=>sequelize.query(`
      SELECT
        t.table_schema as schema,
        t.table_name as table
      FROM information_schema.tables t
      LEFT JOIN information_schema.triggers triggers ON 
        t.table_name = triggers.event_object_table 
        AND t.table_schema = triggers.event_object_schema
        AND triggers.trigger_name = substring(concat($prefix::text, lower(t.table_name), $suffix::text), 0, 64)
      WHERE
        t.table_schema IN ('public', 'logs')
        AND t.table_type != 'VIEW'
        AND triggers.trigger_name IS NULL -- No matching trigger
      GROUP BY t.table_schema, t.table_name -- Group to ensure unique results
    `, {
        type: _sequelize.QueryTypes.SELECT,
        bind: {
            prefix,
            suffix
        }
    }).then((rows)=>rows.map((row)=>({
                schema: row.schema,
                table: row.table
            })).filter(({ schema, table })=>!tableNameMatch(schema, table, excludes)));
const tablesWithTrigger = (sequelize, prefix, suffix, excludes = _constants.NON_SYNCING_TABLES)=>sequelize.query(`
      SELECT
        t.table_schema as schema,
        t.table_name as table
      FROM information_schema.tables t
      JOIN information_schema.triggers triggers ON 
        t.table_name = triggers.event_object_table 
        AND t.table_schema = triggers.event_object_schema
        AND triggers.trigger_name = substring(concat($prefix::text, lower(t.table_name), $suffix::text), 0, 64)
      WHERE
        t.table_schema IN ('public', 'logs')
        AND t.table_type != 'VIEW'
      GROUP BY t.table_schema, t.table_name -- Group to ensure unique results
    `, {
        type: _sequelize.QueryTypes.SELECT,
        bind: {
            prefix,
            suffix
        }
    }).then((rows)=>rows.map((row)=>({
                schema: row.schema,
                table: row.table
            })).filter(({ schema, table })=>!excludes.includes(`${schema}.${table}`)));
async function runPreMigration(log, sequelize) {
    // remove sync tick trigger before migrations
    // migrations are deterministic, so updating the sync tick just creates useless churn
    for (const { schema, table } of (await tablesWithTrigger(sequelize, 'set_', '_updated_at_sync_tick'))){
        log.info(`Removing updated_at_sync_tick trigger from ${schema}.${table}`);
        await sequelize.query(`DROP TRIGGER set_${table}_updated_at_sync_tick ON "${schema}"."${table}"`);
    }
    // remove changelog trigger before migrations
    for (const { schema, table } of (await tablesWithTrigger(sequelize, 'record_', '_changelog'))){
        log.info(`Removing changelog trigger from ${schema}.${table}`);
        await sequelize.query(`DROP TRIGGER record_${table}_changelog ON "${schema}"."${table}"`);
    }
}
async function runPostMigration(log, sequelize) {
    // add column: holds last update tick, default to 0 (will be caught in any initial sync) on central server
    // and SYNC_TICK_FLAGS.UPDATED_ELSEWHERE (not marked for sync) on facility
    // triggers will overwrite the default for future data, but this works for existing data
    const isFacilityServer = !!(0, _selectFacilityIds.selectFacilityIds)(_config.default);
    const initialValue = isFacilityServer ? _constants1.SYNC_TICK_FLAGS.LAST_UPDATED_ELSEWHERE : 0;
    for (const { schema, table } of (await tablesWithoutColumn(sequelize, 'updated_at_sync_tick'))){
        log.info(`Adding updated_at_sync_tick column to ${schema}.${table}`);
        await sequelize.query(`
      ALTER TABLE "${schema}"."${table}" ADD COLUMN updated_at_sync_tick BIGINT NOT NULL DEFAULT ${initialValue};
    `);
        await sequelize.query(`
      CREATE INDEX ${table}_updated_at_sync_tick_index ON "${schema}"."${table}" (updated_at_sync_tick);
    `);
    }
    const functionExists = (name)=>sequelize.query('select count(*) as count from pg_catalog.pg_proc where proname = $name', {
            type: _sequelize.QueryTypes.SELECT,
            bind: {
                name
            }
        }).then((rows)=>rows?.[0]?.count > 0);
    // add trigger: before insert or update, set updated_at (overriding any that is passed in)
    if (await functionExists('set_updated_at_sync_tick')) {
        for (const { schema, table } of (await tablesWithoutTrigger(sequelize, 'set_', '_updated_at_sync_tick'))){
            log.info(`Adding updated_at_sync_tick trigger to ${schema}.${table}`);
            await sequelize.query(`
      CREATE TRIGGER set_${table}_updated_at_sync_tick
      BEFORE INSERT OR UPDATE ON "${schema}"."${table}"
      FOR EACH ROW
      EXECUTE FUNCTION public.set_updated_at_sync_tick();
    `);
        }
    }
    // add trigger to table for pg notify
    if (await functionExists('notify_table_changed')) {
        for (const { schema, table } of (await tablesWithoutTrigger(sequelize, 'notify_', '_changed'))){
            log.info(`Adding notify change trigger to ${schema}.${table}`);
            await sequelize.query(`
      CREATE TRIGGER notify_${table}_changed
      AFTER INSERT OR UPDATE OR DELETE ON "${schema}"."${table}"
      FOR EACH ROW
      EXECUTE FUNCTION public.notify_table_changed();
    `);
        }
    }
    // add trigger to table for changelogs
    if (await functionExists('record_change')) {
        for (const { schema, table } of (await tablesWithoutTrigger(sequelize, 'record_', '_changelog', _constants.NON_LOGGED_TABLES))){
            log.info(`Adding changelog trigger to ${schema}.${table}`);
            await sequelize.query(`
      CREATE TRIGGER record_${table}_changelog
      AFTER INSERT OR UPDATE ON "${schema}"."${table}"
      FOR EACH ROW
      EXECUTE FUNCTION logs.record_change();
    `);
        }
    }
}

//# sourceMappingURL=migrationHooks.js.map