"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "SurveyResponseAnswer", {
    enumerable: true,
    get: function() {
        return SurveyResponseAnswer;
    }
});
const _lodash = require("lodash");
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _Model = require("./Model");
const _errors = require("@tamanu/shared/errors");
const _calculations = require("@tamanu/shared/utils/calculations");
const _fields = require("@tamanu/shared/utils/fields");
const _buildPatientLinkedLookupFilter = require("../sync/buildPatientLinkedLookupFilter");
function _define_property(obj, key, value) {
    if (key in obj) {
        Object.defineProperty(obj, key, {
            value: value,
            enumerable: true,
            configurable: true,
            writable: true
        });
    } else {
        obj[key] = value;
    }
    return obj;
}
let SurveyResponseAnswer = class SurveyResponseAnswer extends _Model.Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            name: _sequelize.DataTypes.STRING,
            body: _sequelize.DataTypes.TEXT
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL
        });
    }
    static initRelations(models) {
        this.belongsTo(models.ProgramDataElement, {
            foreignKey: 'dataElementId'
        });
        this.belongsTo(models.SurveyResponse, {
            foreignKey: 'responseId',
            as: 'surveyResponse'
        });
        this.hasMany(models.VitalLog, {
            foreignKey: 'answerId',
            as: 'vitalLog'
        });
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable, sessionConfig) {
        if (patientCount === 0) {
            return null;
        }
        // manually construct "joins", as survey_response join uses a non-conventional join column
        const joins = `
      JOIN survey_responses ON survey_response_answers.response_id = survey_responses.id
      JOIN encounters ON survey_responses.encounter_id = encounters.id
    `;
        // remove answers to sensitive surveys from mobile
        if (sessionConfig.isMobile) {
            return `
        ${joins}
        JOIN surveys ON survey_responses.survey_id = surveys.id
        WHERE
          encounters.patient_id in (SELECT patient_id FROM ${markedForSyncPatientsTable})
        AND
          surveys.is_sensitive = FALSE
        AND
          ${this.tableName}.updated_at_sync_tick > :since
      `;
        }
        return `
      ${joins}
      WHERE
        encounters.patient_id in (SELECT patient_id FROM ${markedForSyncPatientsTable})
      AND
        ${this.tableName}.updated_at_sync_tick > :since
    `;
    }
    static async buildSyncLookupQueryDetails() {
        return {
            select: await (0, _buildPatientLinkedLookupFilter.buildEncounterPatientIdSelect)(this),
            joins: `
        JOIN survey_responses ON survey_response_answers.response_id = survey_responses.id
        JOIN encounters ON survey_responses.encounter_id = encounters.id
      `
        };
    }
    // To be called after creating/updating a vitals survey response answer. Checks if
    // said answer is used in calculated questions and updates them accordingly.
    async upsertCalculatedQuestions(data) {
        if (!this.sequelize.isInsideTransaction()) {
            throw new Error('upsertCalculatedQuestions must always run inside a transaction!');
        }
        const { models } = this.sequelize;
        const surveyResponse = await this.getSurveyResponse();
        const vitalsSurvey = await models.Survey.getVitalsSurvey();
        const isVitalSurvey = surveyResponse.surveyId === vitalsSurvey?.id;
        if (isVitalSurvey === false) {
            throw new _errors.InvalidOperationError('upsertCalculatedQuestions must only be called with vitals answers');
        }
        // Get necessary info and data shapes for running calculations
        const screenComponents = await models.SurveyScreenComponent.getComponentsForSurvey(surveyResponse.surveyId, {
            includeAllVitals: true
        });
        const calculatedScreenComponents = screenComponents.filter((c)=>c.calculation);
        const updatedAnswerDataElement = await this.getProgramDataElement();
        const answers = await surveyResponse.getAnswers();
        const values = {};
        answers.forEach((answer)=>{
            values[answer.dataElementId] = answer.body;
        });
        const calculatedValues = (0, _calculations.runCalculations)(screenComponents, values);
        for (const component of calculatedScreenComponents){
            if (component.calculation.includes(updatedAnswerDataElement.code) === false) {
                continue;
            }
            // Sanitize value
            const stringValue = (0, _fields.getStringValue)(component.dataElement.type, calculatedValues[component.dataElement.id]);
            const newCalculatedValue = stringValue ?? '';
            // Check if the calculated answer was created or not. It might've been missed
            // if no values used in its calculation were registered the first time.
            const existingCalculatedAnswer = answers.find((answer)=>answer.dataElementId === component.dataElement.id);
            const previousCalculatedValue = existingCalculatedAnswer?.body;
            let newCalculatedAnswer = null;
            if (existingCalculatedAnswer) {
                await existingCalculatedAnswer.update({
                    body: newCalculatedValue
                });
            } else {
                newCalculatedAnswer = await models.SurveyResponseAnswer.create({
                    dataElementId: component.dataElement.id,
                    body: newCalculatedValue,
                    responseId: surveyResponse.id
                });
            }
            const { date, reasonForChange, user } = data;
            await models.VitalLog.create({
                date,
                reasonForChange,
                previousValue: previousCalculatedValue || null,
                newValue: newCalculatedValue,
                recordedById: user.id,
                answerId: existingCalculatedAnswer?.id || newCalculatedAnswer?.id
            });
        }
        return this;
    }
};
// eslint-disable-next-line no-unused-vars
_define_property(SurveyResponseAnswer, "getDefaultId", async (resource, settings)=>{
    const { models } = SurveyResponseAnswer.sequelize;
    const code = await settings.get(`survey.defaultCodes.${resource}`);
    const modelName = (0, _lodash.upperFirst)(resource);
    const model = models[modelName];
    if (!model) {
        throw new Error(`Model not found: ${modelName}`);
    }
    const record = await model.findOne({
        where: {
            code
        }
    });
    if (!record) {
        throw new Error(`Could not find default answer for '${resource}': code '${code}' not found (check survey.defaultCodes.${resource} in the settings)`);
    }
    return record.id;
});

//# sourceMappingURL=SurveyResponseAnswer.js.map