import { DataTypes, Sequelize } from 'sequelize';
import { SYNC_DIRECTIONS } from '@tamanu/constants';
import { InvalidOperationError } from '@tamanu/shared/errors';
import { getCurrentDateTimeString } from '@tamanu/utils/dateTime';
import { Model } from './Model';
import { buildEncounterLinkedSyncFilter, buildEncounterLinkedSyncFilterJoins } from '../sync/buildEncounterLinkedSyncFilter';
import { buildEncounterPatientIdSelect } from '../sync/buildPatientLinkedLookupFilter';
import { dateTimeType } from '../types/model';
export class ImagingResult extends Model {
    static initModel(options) {
        super.init({
            id: {
                type: DataTypes.UUID,
                allowNull: false,
                primaryKey: true,
                defaultValue: Sequelize.fn('gen_random_uuid')
            },
            visibilityStatus: {
                type: DataTypes.STRING,
                allowNull: false,
                defaultValue: 'current'
            },
            completedAt: dateTimeType('completedAt', {
                allowNull: false,
                defaultValue: getCurrentDateTimeString
            }),
            description: {
                type: DataTypes.TEXT
            },
            externalCode: DataTypes.TEXT,
            resultImageUrl: {
                type: DataTypes.TEXT,
                defaultValue: null,
                allowNull: true
            }
        }, {
            ...options,
            syncDirection: SYNC_DIRECTIONS.BIDIRECTIONAL,
            validate: {
                mustHaveImagingRequest () {
                    if (!this.imagingRequestId) {
                        throw new InvalidOperationError('An imaging result must be associated with an imaging request.');
                    }
                }
            }
        });
    }
    static getListReferenceAssociations() {
        return [
            'request',
            'completedBy'
        ];
    }
    static initRelations(models) {
        this.belongsTo(models.ImagingRequest, {
            foreignKey: 'imagingRequestId',
            as: 'request'
        });
        this.belongsTo(models.User, {
            foreignKey: 'completedById',
            as: 'completedBy'
        });
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return buildEncounterLinkedSyncFilter([
            this.tableName,
            'imaging_requests',
            'encounters'
        ], markedForSyncPatientsTable);
    }
    static async buildSyncLookupQueryDetails() {
        return {
            select: await buildEncounterPatientIdSelect(this),
            joins: buildEncounterLinkedSyncFilterJoins([
                this.tableName,
                'imaging_requests',
                'encounters'
            ])
        };
    }
}

//# sourceMappingURL=ImagingResult.js.map