"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "DocumentMetadata", {
    enumerable: true,
    get: function() {
        return DocumentMetadata;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _dateTime = require("@tamanu/utils/dateTime");
const _Model = require("./Model");
const _buildEncounterLinkedSyncFilter = require("../sync/buildEncounterLinkedSyncFilter");
const _buildSyncLookupSelect = require("../sync/buildSyncLookupSelect");
const _model = require("../types/model");
let DocumentMetadata = class DocumentMetadata extends _Model.Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            name: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: false
            },
            type: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: false
            },
            source: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false,
                defaultValue: _constants.DOCUMENT_SOURCES.UPLOADED
            },
            documentCreatedAt: (0, _model.dateTimeType)('documentCreatedAt'),
            documentUploadedAt: (0, _model.dateTimeType)('documentUploadedAt', {
                allowNull: false,
                defaultValue: _dateTime.getCurrentDateTimeString
            }),
            documentOwner: _sequelize.DataTypes.TEXT,
            note: _sequelize.DataTypes.STRING,
            // Relation can't be managed by sequelize because the
            // attachment won't get downloaded to facility server
            attachmentId: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL
        });
    }
    static initRelations(models) {
        this.belongsTo(models.Encounter, {
            foreignKey: 'encounterId',
            as: 'encounter'
        });
        this.belongsTo(models.Patient, {
            foreignKey: 'patientId',
            as: 'patient'
        });
        this.belongsTo(models.Department, {
            foreignKey: 'departmentId',
            as: 'department'
        });
    }
    static getListReferenceAssociations() {
        return [
            'department'
        ];
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        const join = (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilterJoins)([
            this.tableName,
            'encounters'
        ]);
        return `
      ${join}
      WHERE (
        encounters.patient_id IN (SELECT patient_id FROM ${markedForSyncPatientsTable})
        OR
        ${this.tableName}.patient_id IN (SELECT patient_id FROM ${markedForSyncPatientsTable})
      )
      AND ${this.tableName}.updated_at_sync_tick > :since
    `;
    }
    static async buildSyncLookupQueryDetails() {
        return {
            select: await (0, _buildSyncLookupSelect.buildSyncLookupSelect)(this, {
                patientId: 'COALESCE(document_metadata.patient_id, encounters.patient_id)'
            }),
            joins: `
        LEFT JOIN encounters ON ${this.tableName}.encounter_id = encounters.id
        LEFT JOIN patients ON ${this.tableName}.patient_id = encounters.id
      `
        };
    }
};

//# sourceMappingURL=DocumentMetadata.js.map