import { DataTypes } from 'sequelize';
import { NOTIFICATION_TYPES, SYNC_DIRECTIONS } from '@tamanu/constants';
import { getCurrentDateTimeString } from '@tamanu/utils/dateTime';
import { Model } from './Model';
import { dateTimeType } from '../types/model';
export class Prescription extends Model {
    static initModel({ primaryKey, ...options }, models) {
        super.init({
            id: primaryKey,
            isOngoing: DataTypes.BOOLEAN,
            isPrn: DataTypes.BOOLEAN,
            isVariableDose: DataTypes.BOOLEAN,
            doseAmount: DataTypes.DECIMAL,
            units: {
                type: DataTypes.STRING,
                allowNull: false
            },
            frequency: {
                type: DataTypes.STRING,
                allowNull: false
            },
            idealTimes: DataTypes.ARRAY(DataTypes.STRING),
            route: {
                type: DataTypes.STRING,
                allowNull: false
            },
            date: dateTimeType('date', {
                allowNull: false,
                defaultValue: getCurrentDateTimeString
            }),
            startDate: dateTimeType('startDate', {
                allowNull: false,
                defaultValue: getCurrentDateTimeString
            }),
            endDate: dateTimeType('endDate'),
            durationValue: DataTypes.DECIMAL,
            durationUnit: DataTypes.STRING,
            indication: DataTypes.STRING,
            isPhoneOrder: DataTypes.BOOLEAN,
            notes: DataTypes.STRING,
            pharmacyNotes: DataTypes.STRING,
            displayPharmacyNotesInMar: DataTypes.BOOLEAN,
            quantity: DataTypes.INTEGER,
            discontinued: DataTypes.BOOLEAN,
            discontinuedDate: DataTypes.STRING,
            discontinuingReason: DataTypes.STRING,
            repeats: DataTypes.INTEGER
        }, {
            ...options,
            syncDirection: SYNC_DIRECTIONS.BIDIRECTIONAL,
            hooks: {
                afterCreate: async (prescription)=>{
                    if (prescription.durationValue && prescription.durationUnit) {
                        const { add } = await import('date-fns');
                        prescription.endDate = add(new Date(prescription.startDate), {
                            [prescription.durationUnit]: prescription.durationValue
                        }).toISOString();
                    }
                    await prescription.save();
                },
                afterUpdate: async (prescription, options)=>{
                    if (prescription.changed('pharmacyNotes')) {
                        await models.Notification.pushNotification(NOTIFICATION_TYPES.PHARMACY_NOTE, prescription.dataValues, {
                            transaction: options.transaction
                        });
                    }
                    if (prescription.changed('discontinued') && prescription.discontinued) {
                        await models.MedicationAdministrationRecord.removeInvalidMedicationAdministrationRecords(options.transaction);
                    }
                }
            }
        });
    }
    static initRelations(models) {
        this.belongsTo(models.User, {
            foreignKey: 'prescriberId',
            as: 'prescriber'
        });
        this.belongsTo(models.User, {
            foreignKey: 'discontinuingClinicianId',
            as: 'discontinuingClinician'
        });
        this.hasOne(models.EncounterPrescription, {
            foreignKey: 'prescriptionId',
            as: 'encounterPrescription'
        });
        this.belongsToMany(models.Encounter, {
            through: models.EncounterPrescription,
            foreignKey: 'prescriptionId',
            as: 'encounters'
        });
        this.hasOne(models.PatientOngoingPrescription, {
            foreignKey: 'prescriptionId',
            as: 'patientOngoingPrescription'
        });
        this.belongsToMany(models.Patient, {
            through: models.PatientOngoingPrescription,
            foreignKey: 'prescriptionId',
            as: 'patients'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'medicationId',
            as: 'medication'
        });
        this.hasMany(models.MedicationAdministrationRecord, {
            foreignKey: 'prescriptionId',
            as: 'medicationAdministrationRecords'
        });
    }
    static getListReferenceAssociations() {
        return [
            'medication',
            'prescriber',
            'discontinuingClinician'
        ];
    }
    static buildSyncFilter() {
        return null; // syncs everywhere
    }
    static buildSyncLookupQueryDetails() {
        return null; // syncs everywhere
    }
}

//# sourceMappingURL=Prescription.js.map