"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "AutomaticLabTestResultPublisher", {
    enumerable: true,
    get: function() {
        return AutomaticLabTestResultPublisher;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _constants = require("@tamanu/constants");
const _tasks = require("@tamanu/shared/tasks");
const _logging = require("@tamanu/shared/services/logging");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let AutomaticLabTestResultPublisher = class AutomaticLabTestResultPublisher extends _tasks.ScheduledTask {
    getName() {
        return 'AutomaticLabTestResultPublisher';
    }
    async run() {
        // get relevant ids from config
        const labTestIds = Object.keys(this.results);
        // get all pending lab tests with a relevant id
        const tests = await this.models.LabTest.findAll({
            where: {
                result: '',
                labTestTypeId: labTestIds,
                '$labRequest.status$': _constants.LAB_REQUEST_STATUSES.RECEPTION_PENDING
            },
            include: [
                'labTestType',
                'labRequest'
            ],
            limit: this.limit
        });
        this.lastRunCount = tests.length;
        if (this.lastRunCount === 0) {
            _logging.log.info('No lab tests to publish.');
            return;
        }
        _logging.log.info(`Auto-publishing ${this.lastRunCount} lab tests...`);
        for (const test of tests){
            const { labRequest, labTestType } = test;
            try {
                // transaction just exists on any model, nothing specific to LabTest happening on this line
                await this.models.LabTest.sequelize.transaction(async ()=>{
                    // get the appropriate result info for this test
                    const resultData = this.results[labTestType.id];
                    // update test with result + method ID
                    await test.update({
                        labTestMethodId: resultData.labTestMethodId,
                        result: resultData.result,
                        completedDate: labRequest.requestedDate
                    });
                    // publish the lab request (where it will be picked up by certificate notification if relevant)
                    await labRequest.update({
                        status: _constants.LAB_REQUEST_STATUSES.PUBLISHED
                    });
                    // Create a log entry for the lab request status
                    await this.models.LabRequestLog.create({
                        status: labRequest.status,
                        labRequestId: labRequest.id,
                        updatedById: labRequest.requestedById
                    });
                    _logging.log.info(`Auto-published lab request ${labRequest.id} (${labRequest.displayId})`);
                });
            } catch (e) {
                _logging.log.error(`Couldn't auto-publish lab request ${labRequest.id} (${labRequest.displayId})`, e);
            }
        }
    }
    constructor(context, overrideConfig = null){
        const { schedule, results, jitterTime, enabled } = overrideConfig || _config.default.schedules.automaticLabTestResultPublisher;
        super(schedule, _logging.log, jitterTime, enabled);
        this.results = results;
        this.limit = _config.default.limit;
        this.models = context.store.models;
        this.lastRunCount = 0;
    }
};

//# sourceMappingURL=AutomaticLabTestResultPublisher.js.map