"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "DeceasedPatientDischarger", {
    enumerable: true,
    get: function() {
        return DeceasedPatientDischarger;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _sequelize = require("sequelize");
const _tasks = require("@tamanu/shared/tasks");
const _logging = require("@tamanu/shared/services/logging");
const _sleepAsync = require("@tamanu/utils/sleepAsync");
const _errors = require("@tamanu/shared/errors");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let DeceasedPatientDischarger = class DeceasedPatientDischarger extends _tasks.ScheduledTask {
    getName() {
        return 'DeceasedPatientDischarger';
    }
    async run() {
        const { Encounter, Patient } = this.models;
        const query = {
            where: {
                endDate: null,
                '$patient.date_of_death$': {
                    [_sequelize.Op.not]: null
                }
            },
            include: [
                {
                    model: Patient,
                    as: 'patient'
                }
            ]
        };
        const toProcess = await Encounter.count(query);
        if (toProcess === 0) return;
        const { batchSize, batchSleepAsyncDurationInMilliseconds } = this.config;
        // Make sure these exist, else they will prevent the script from working
        if (!batchSize || !batchSleepAsyncDurationInMilliseconds) {
            throw new _errors.InvalidConfigError('batchSize and batchSleepAsyncDurationInMilliseconds must be set for DeceasedPatientDischarger');
        }
        const batchCount = Math.ceil(toProcess / batchSize);
        _logging.log.info(`Auto-discharging ${toProcess} encounters for deceased patients in ${batchCount} batches (${batchSize} records per batch)`);
        for(let i = 0; i < batchCount; i++){
            const encounters = await Encounter.findAll({
                ...query,
                limit: batchSize
            });
            for (const encounter of encounters){
                const patient = await encounter.getPatient();
                if (!patient.dateOfDeath) {
                    _logging.log.warn(`Deceased patient ${patient.id} not actually deceased! Skipping...`);
                    continue;
                }
                const [patientDeathData] = await patient.getDeathData();
                if (!patientDeathData) {
                    _logging.log.warn(`Deceased patient ${patient.id} has no death data! Skipping...`);
                    continue;
                }
                const discharger = await patientDeathData.getClinician();
                await encounter.update({
                    endDate: patient.dateOfDeath,
                    systemNote: 'Automatically discharged',
                    discharge: {
                        dischargerId: discharger.id,
                        note: 'Automatically discharged by deceased patient discharger'
                    }
                });
                _logging.log.info(`Auto-closed encounter with id ${encounter.id} (discharger=${discharger.id}, dod=${patient.dateOfDeath})`);
            }
            await (0, _sleepAsync.sleepAsync)(batchSleepAsyncDurationInMilliseconds);
        }
    }
    constructor(context){
        const conf = _config.default.schedules.deceasedPatientDischarger;
        const { schedule, jitterTime, enabled } = conf;
        super(schedule, _logging.log, jitterTime, enabled);
        this.config = conf;
        this.models = context.store.models;
    }
};

//# sourceMappingURL=DeceasedPatientDischarger.js.map