import { DataTypes, QueryTypes } from 'sequelize';
import { SYNC_DIRECTIONS } from '@tamanu/constants';
import { InvalidOperationError } from '@tamanu/shared/errors';
import { Model } from './Model';
import { buildEncounterLinkedSyncFilter } from '../sync/buildEncounterLinkedSyncFilter';
import { buildEncounterLinkedLookupFilter } from '../sync/buildEncounterLinkedLookupFilter';
import { Facility } from './Facility';
import { log } from '@tamanu/shared/services/logging';
export class Discharge extends Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            note: {
                type: DataTypes.TEXT,
                allowNull: true
            },
            facilityName: {
                type: DataTypes.STRING,
                allowNull: true
            },
            facilityAddress: {
                type: DataTypes.STRING,
                allowNull: true
            },
            facilityTown: {
                type: DataTypes.STRING,
                allowNull: true
            }
        }, {
            ...options,
            syncDirection: SYNC_DIRECTIONS.BIDIRECTIONAL,
            validate: {
                mustHaveEncounter () {
                    if (!this.deletedAt && !this.encounterId) {
                        throw new InvalidOperationError('A discharge must have an encounter.');
                    }
                }
            }
        });
    }
    async address() {
        const encounterFacility = await this.sequelize.query(`
        SELECT f.* FROM encounters e
        JOIN locations l on l.id = e.location_id
        JOIN facilities f on f.id = l.facility_id
        WHERE e.id = $encounterId
        `, {
            type: QueryTypes.SELECT,
            model: Facility,
            mapToModel: true,
            bind: {
                encounterId: this.encounterId
            }
        }).then((res)=>res?.[0], (err)=>{
            log.warn('Failed to fetch encounter facility', err);
            return null;
        });
        return {
            name: encounterFacility?.name ?? this.facilityName,
            address: encounterFacility?.streetAddress ?? this.facilityAddress,
            town: encounterFacility?.cityTown ?? this.facilityTown
        };
    }
    static getFullReferenceAssociations() {
        return [
            'discharger',
            'disposition'
        ];
    }
    static initRelations(models) {
        this.belongsTo(models.Encounter, {
            foreignKey: 'encounterId',
            as: 'encounter'
        });
        this.belongsTo(models.User, {
            foreignKey: 'dischargerId',
            as: 'discharger'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'dispositionId',
            as: 'disposition'
        });
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return buildEncounterLinkedSyncFilter([
            this.tableName,
            'encounters'
        ], markedForSyncPatientsTable);
    }
    static async buildSyncLookupQueryDetails() {
        return buildEncounterLinkedLookupFilter(this);
    }
}

//# sourceMappingURL=Discharge.js.map