"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "PatientProgramRegistration", {
    enumerable: true,
    get: function() {
        return PatientProgramRegistration;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _dateTime = require("@tamanu/utils/dateTime");
const _Model = require("./Model");
const _model = require("../types/model");
let PatientProgramRegistration = class PatientProgramRegistration extends _Model.Model {
    static initModel(options) {
        super.init({
            id: {
                // patient_program_registration records use a generated primary key that enforces
                // one per patient and program registry, even across a distributed sync system
                type: `TEXT GENERATED ALWAYS AS (REPLACE("patient_id", ';', ':') || ';' || REPLACE("program_registry_id", ';', ':')) STORED`,
                set () {
                // any sets of the convenience generated "id" field can be ignored
                }
            },
            patientId: {
                type: _sequelize.DataTypes.STRING,
                primaryKey: true,
                references: {
                    model: 'patients',
                    key: 'id'
                }
            },
            programRegistryId: {
                type: _sequelize.DataTypes.STRING,
                primaryKey: true,
                references: {
                    model: 'program_registries',
                    key: 'id'
                }
            },
            date: (0, _model.dateTimeType)('date', {
                allowNull: false,
                defaultValue: _dateTime.getCurrentDateTimeString
            }),
            registrationStatus: {
                allowNull: false,
                type: _sequelize.DataTypes.TEXT,
                defaultValue: _constants.REGISTRATION_STATUSES.ACTIVE
            },
            deactivatedDate: (0, _model.dateTimeType)('deactivatedDate', {
                allowNull: true
            })
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL
        });
    }
    static getFullReferenceAssociations() {
        return [
            'programRegistry',
            'clinicalStatus',
            'clinician',
            'deactivatedClinician',
            'registeringFacility',
            'facility',
            'village'
        ];
    }
    static getListReferenceAssociations() {
        return [
            'clinicalStatus',
            'clinician',
            'deactivatedClinician'
        ];
    }
    static initRelations(models) {
        this.belongsTo(models.Patient, {
            foreignKey: {
                name: 'patientId',
                allowNull: false
            },
            as: 'patient'
        });
        this.belongsTo(models.ProgramRegistry, {
            foreignKey: {
                name: 'programRegistryId',
                allowNull: false
            },
            as: 'programRegistry'
        });
        this.belongsTo(models.ProgramRegistryClinicalStatus, {
            foreignKey: 'clinicalStatusId',
            as: 'clinicalStatus'
        });
        this.belongsTo(models.User, {
            foreignKey: {
                name: 'clinicianId',
                allowNull: false
            },
            as: 'clinician'
        });
        this.belongsTo(models.User, {
            foreignKey: 'deactivatedClinicianId',
            as: 'deactivatedClinician'
        });
        this.belongsTo(models.Facility, {
            foreignKey: 'registeringFacilityId',
            as: 'registeringFacility'
        });
        // 1. Note that only one of facilityId or villageId will usually be set,
        // depending on the currentlyAtType of the related programRegistry.
        this.belongsTo(models.Facility, {
            foreignKey: 'facilityId',
            as: 'facility'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'villageId',
            as: 'village'
        });
        this.hasMany(models.PatientProgramRegistrationCondition, {
            foreignKey: 'patientProgramRegistrationId',
            as: 'conditions'
        });
    }
    static async getRegistrationsForPatient(patientId) {
        return this.sequelize.models.PatientProgramRegistration.findAll({
            where: {
                registrationStatus: {
                    [_sequelize.Op.ne]: _constants.REGISTRATION_STATUSES.RECORDED_IN_ERROR
                },
                patientId
            },
            include: [
                'clinicalStatus',
                'programRegistry',
                'deactivatedClinician'
            ],
            order: [
                // "active" > "removed"
                [
                    'registrationStatus',
                    'ASC'
                ],
                [
                    _sequelize.Sequelize.col('programRegistry.name'),
                    'ASC'
                ]
            ]
        });
    }
    static buildSyncFilter() {
        return null;
    }
    static buildSyncLookupQueryDetails() {
        return null; // syncs everywhere
    }
};

//# sourceMappingURL=PatientProgramRegistration.js.map