import { Document, StyleSheet, View } from '@react-pdf/renderer';
import React from 'react';
import { CertificateHeader, styles } from './Layout';
import { PatientDetailsWithAddress } from './printComponents/PatientDetailsWithAddress';
import { DIAGNOSIS_CERTAINTIES_TO_HIDE, DRUG_ROUTE_LABELS } from '@tamanu/constants';
import { EncounterDetailsExtended } from './printComponents/EncounterDetailsExtended';
import { P } from './Typography';
import { LetterheadSection } from './LetterheadSection';
import { useLanguageContext, withLanguageContext } from '../pdf/languageContext';
import { Page } from '../pdf/Page';
import { Text } from '../pdf/Text';
import { Table } from './Table';
import { getMedicationDoseDisplay, getTranslatedFrequency } from '../medication';
const borderStyle = '1 solid black';
const tableLabelWidth = 150;
const tablePadding = 10;
const generalStyles = StyleSheet.create({
    tableContainer: {
        marginVertical: 5
    },
    sectionContainer: {
        marginVertical: 7
    }
});
const TableContainer = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: generalStyles.tableContainer,
        ...props
    });
const SectionContainer = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: generalStyles.sectionContainer,
        ...props
    });
const infoBoxStyles = StyleSheet.create({
    row: {
        flexDirection: 'row',
        border: borderStyle
    },
    labelCol: {
        borderRight: borderStyle,
        width: tableLabelWidth,
        padding: tablePadding
    },
    dataCol: {
        flex: 1,
        padding: tablePadding
    },
    labelText: {
        fontSize: 10
    },
    infoText: {
        fontSize: 10
    }
});
const InfoBoxRow = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: infoBoxStyles.row,
        ...props
    });
const InfoBoxLabelCol = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: infoBoxStyles.labelCol,
        ...props
    });
const InfoBoxDataCol = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: infoBoxStyles.dataCol,
        ...props
    });
const notesSectionStyles = StyleSheet.create({
    notesBox: {
        border: borderStyle,
        minHeight: 76,
        padding: 10
    }
});
const extractOngoingConditions = (patientConditions)=>patientConditions.map((item)=>item?.diagnosis?.name);
const extractDiagnosesInfo = ({ diagnoses, getSetting })=>{
    const displayIcd10Codes = getSetting('features.displayIcd10CodesInDischargeSummary');
    if (!displayIcd10Codes) {
        return diagnoses.map((item)=>item?.diagnosis?.name);
    } else {
        return diagnoses.map((item)=>`${item?.diagnosis?.name} (${item?.diagnosis?.code})`);
    }
};
const extractProceduresInfo = ({ procedures, getSetting })=>{
    const displayProcedureCodes = getSetting('features.displayProcedureCodesInDischargeSummary');
    if (!displayProcedureCodes) {
        return procedures.map((item)=>item?.procedureType?.name);
    } else {
        return procedures.map((item)=>`${item?.procedureType?.name} (${item?.procedureType?.code})`);
    }
};
const InfoBox = ({ label, info })=>/*#__PURE__*/ React.createElement(InfoBoxRow, null, /*#__PURE__*/ React.createElement(InfoBoxLabelCol, null, /*#__PURE__*/ React.createElement(Text, {
        bold: true,
        style: infoBoxStyles.labelText
    }, label)), /*#__PURE__*/ React.createElement(InfoBoxDataCol, null, info.map((item, index)=>{
        return /*#__PURE__*/ React.createElement(Text, {
            style: infoBoxStyles.infoText,
            key: index
        }, item);
    })));
const DiagnosesTable = ({ title, diagnoses, getSetting })=>/*#__PURE__*/ React.createElement(InfoBox, {
        label: title,
        info: extractDiagnosesInfo({
            diagnoses,
            getSetting
        })
    });
const ProceduresTable = ({ procedures, getSetting })=>/*#__PURE__*/ React.createElement(InfoBox, {
        label: "Procedures",
        info: extractProceduresInfo({
            procedures,
            getSetting
        })
    });
const NotesSection = ({ notes })=>/*#__PURE__*/ React.createElement(View, null, /*#__PURE__*/ React.createElement(P, {
        bold: true,
        fontSize: 11,
        mb: 3
    }, "Discharge planning notes"), /*#__PURE__*/ React.createElement(View, {
        style: notesSectionStyles.notesBox
    }, /*#__PURE__*/ React.createElement(Text, {
        style: infoBoxStyles.infoText
    }, notes)));
const columns = (getTranslation, getEnumTranslation)=>[
        {
            key: 'medication',
            title: getTranslation('pdf.table.column.medication', 'Medication'),
            accessor: ({ medication, notes })=>/*#__PURE__*/ React.createElement(View, null, /*#__PURE__*/ React.createElement(Text, null, medication?.name + `\n`), /*#__PURE__*/ React.createElement(Text, {
                    style: {
                        fontFamily: 'Helvetica-Oblique'
                    }
                }, notes)),
            customStyles: {
                minWidth: 180
            }
        },
        {
            key: 'dose',
            title: getTranslation('pdf.table.column.dose', 'Dose'),
            accessor: (medication)=>{
                return /*#__PURE__*/ React.createElement(Text, null, getMedicationDoseDisplay(medication, getTranslation, getEnumTranslation), medication?.isPrn && ` ${getTranslation('medication.table.prn', 'PRN')}`);
            }
        },
        {
            key: 'frequency',
            title: getTranslation('pdf.table.column.frequency', 'Frequency'),
            accessor: ({ frequency })=>getTranslatedFrequency(frequency, getTranslation),
            customStyles: {
                minWidth: 30
            }
        },
        {
            key: 'route',
            title: getTranslation('pdf.table.column.route', 'Route'),
            accessor: ({ route })=>getEnumTranslation(DRUG_ROUTE_LABELS, route)
        },
        {
            key: 'quantity',
            title: getTranslation('pdf.table.column.quantity', 'Quantity'),
            accessor: ({ quantity })=>quantity
        },
        {
            key: 'repeats',
            title: getTranslation('pdf.table.column.repeats', 'Repeats'),
            accessor: ({ repeats })=>repeats || 0
        }
    ];
const DischargeSummaryPrintoutComponent = ({ patientData, encounter, discharge, patientConditions, certificateData, getLocalisation, getSetting })=>{
    const { getTranslation, getEnumTranslation } = useLanguageContext();
    const { logo } = certificateData;
    const { diagnoses, procedures, medications } = encounter;
    const visibleMedications = medications.filter((m)=>m.encounterPrescription?.isSelectedForDischarge).sort((a, b)=>a.medication.name.localeCompare(b.medication.name));
    const visibleDiagnoses = diagnoses.filter(({ certainty })=>!DIAGNOSIS_CERTAINTIES_TO_HIDE.includes(certainty));
    const primaryDiagnoses = visibleDiagnoses.filter((d)=>d.isPrimary);
    const secondaryDiagnoses = visibleDiagnoses.filter((d)=>!d.isPrimary);
    const notes = discharge?.note;
    const { name: facilityName, address: facilityAddress, town: facilityTown } = discharge.address;
    // change header if facility details are present in discharge
    if (facilityName && facilityAddress && certificateData?.title) {
        certificateData = {
            ...certificateData,
            title: facilityName,
            subTitle: facilityTown ? `${facilityAddress}, ${facilityTown}` : facilityAddress
        };
    }
    return /*#__PURE__*/ React.createElement(Document, null, /*#__PURE__*/ React.createElement(Page, {
        size: "A4",
        style: styles.page
    }, /*#__PURE__*/ React.createElement(CertificateHeader, null, /*#__PURE__*/ React.createElement(LetterheadSection, {
        certificateTitle: "Patient discharge summary",
        letterheadConfig: certificateData,
        logoSrc: logo
    })), /*#__PURE__*/ React.createElement(SectionContainer, null, /*#__PURE__*/ React.createElement(PatientDetailsWithAddress, {
        patient: patientData,
        getLocalisation: getLocalisation,
        getSetting: getSetting
    })), /*#__PURE__*/ React.createElement(SectionContainer, null, /*#__PURE__*/ React.createElement(EncounterDetailsExtended, {
        encounter: encounter,
        discharge: discharge
    })), /*#__PURE__*/ React.createElement(SectionContainer, null, patientConditions.length > 0 && /*#__PURE__*/ React.createElement(TableContainer, null, /*#__PURE__*/ React.createElement(InfoBox, {
        label: "Ongoing conditions",
        info: extractOngoingConditions(patientConditions)
    })), primaryDiagnoses.length > 0 && /*#__PURE__*/ React.createElement(TableContainer, null, /*#__PURE__*/ React.createElement(DiagnosesTable, {
        title: "Primary diagnoses",
        diagnoses: primaryDiagnoses,
        getSetting: getSetting
    })), secondaryDiagnoses.length > 0 && /*#__PURE__*/ React.createElement(TableContainer, null, /*#__PURE__*/ React.createElement(DiagnosesTable, {
        title: "Secondary diagnoses",
        diagnoses: secondaryDiagnoses,
        getSetting: getSetting
    })), procedures.length > 0 && /*#__PURE__*/ React.createElement(TableContainer, null, /*#__PURE__*/ React.createElement(ProceduresTable, {
        procedures: procedures,
        getSetting: getSetting
    })), visibleMedications.length > 0 && /*#__PURE__*/ React.createElement(TableContainer, null, /*#__PURE__*/ React.createElement(Table, {
        data: visibleMedications,
        columns: columns(getTranslation, getEnumTranslation),
        getLocalisation: getLocalisation,
        getSetting: getSetting,
        columnStyle: {
            padding: '10px 5px'
        }
    }))), /*#__PURE__*/ React.createElement(SectionContainer, null, /*#__PURE__*/ React.createElement(NotesSection, {
        notes: notes
    }))));
};
export const DischargeSummaryPrintout = withLanguageContext(DischargeSummaryPrintoutComponent);

//# sourceMappingURL=DischargeSummaryPrintout.js.map