import { type Transaction } from 'sequelize';
import { Model } from './Model';
import { type InitOptions, type Models } from '../types/model';
import type { Prescription } from './Prescription';
export declare class MedicationAdministrationRecord extends Model {
    id: string;
    status?: string;
    dueAt: string;
    recordedAt?: string;
    isAutoGenerated?: boolean;
    prescriptionId?: string;
    changingStatusReason?: string;
    reasonNotGivenId?: string;
    changingNotGivenInfoReason?: string;
    recordedByUserId?: string;
    isEdited?: boolean;
    isError?: boolean;
    errorNotes?: string;
    static initModel({ primaryKey, ...options }: InitOptions, models: Models): void;
    /**
     * Generates upcoming Medication Administration Records (MARs) for a given prescription.
     *
     * This function calculates and creates MAR entries based on the prescription's
     * frequency, start date, end date, ideal administration times, and discontinuation date.
     * It ensures that MARs are generated up to a configurable timeframe in the future
     * (defaulting to 72 hours) or until the prescription's end date, whichever is earlier.
     *
     * Special handling is included for 'IMMEDIATELY' and 'AS_DIRECTED' frequencies,
     * as well as for calculating next due dates for various frequencies like
     * 'EVERY_SECOND_DAY', 'ONCE_A_WEEK', and 'ONCE_A_MONTH'.
     *
     * It avoids creating duplicate records by checking the last existing MAR and
     * skips generation if the calculated due date falls outside the valid prescription period
     * or before the last generated MAR.
     *
     * IMPORTANT: keep it in sync with the mobile app's MedicationAdministrationRecord model
     *
     * @param prescription The prescription object for which to generate MARs.
     */
    static generateMedicationAdministrationRecords(prescription: Prescription): Promise<void>;
    /**
     * Removes medication administration records that are no longer valid due to:
     * 1. The encounter being discharged (MARs with due dates after the encounter end date)
     * 2. The prescription being discontinued (MARs with due dates after the prescription discontinued date)
     *
     * @param {Transaction} [transaction] - Optional transaction to use for the database operations
     */
    static removeInvalidMedicationAdministrationRecords(transaction?: Transaction | null): Promise<void>;
    static getListReferenceAssociations(): string[];
    static initRelations(models: Models): void;
    static buildSyncFilter(): null;
    static buildSyncLookupQueryDetails(): null;
}
//# sourceMappingURL=MedicationAdministrationRecord.d.ts.map