"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "Prescription", {
    enumerable: true,
    get: function() {
        return Prescription;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _dateTime = require("@tamanu/utils/dateTime");
const _Model = require("./Model");
const _model = require("../types/model");
const _buildEncounterLinkedLookupFilter = require("../sync/buildEncounterLinkedLookupFilter");
function _getRequireWildcardCache(nodeInterop) {
    if (typeof WeakMap !== "function") return null;
    var cacheBabelInterop = new WeakMap();
    var cacheNodeInterop = new WeakMap();
    return (_getRequireWildcardCache = function(nodeInterop) {
        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
    })(nodeInterop);
}
function _interop_require_wildcard(obj, nodeInterop) {
    if (!nodeInterop && obj && obj.__esModule) {
        return obj;
    }
    if (obj === null || typeof obj !== "object" && typeof obj !== "function") {
        return {
            default: obj
        };
    }
    var cache = _getRequireWildcardCache(nodeInterop);
    if (cache && cache.has(obj)) {
        return cache.get(obj);
    }
    var newObj = {
        __proto__: null
    };
    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
    for(var key in obj){
        if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) {
            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
            if (desc && (desc.get || desc.set)) {
                Object.defineProperty(newObj, key, desc);
            } else {
                newObj[key] = obj[key];
            }
        }
    }
    newObj.default = obj;
    if (cache) {
        cache.set(obj, newObj);
    }
    return newObj;
}
let Prescription = class Prescription extends _Model.Model {
    static initModel({ primaryKey, ...options }, models) {
        super.init({
            id: primaryKey,
            isOngoing: _sequelize.DataTypes.BOOLEAN,
            isPrn: _sequelize.DataTypes.BOOLEAN,
            isVariableDose: _sequelize.DataTypes.BOOLEAN,
            doseAmount: _sequelize.DataTypes.DECIMAL,
            units: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false
            },
            frequency: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false
            },
            idealTimes: _sequelize.DataTypes.ARRAY(_sequelize.DataTypes.STRING),
            route: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false
            },
            date: (0, _model.dateTimeType)('date', {
                allowNull: false,
                defaultValue: _dateTime.getCurrentDateTimeString
            }),
            startDate: (0, _model.dateTimeType)('startDate', {
                allowNull: false,
                defaultValue: _dateTime.getCurrentDateTimeString
            }),
            endDate: (0, _model.dateTimeType)('endDate'),
            durationValue: _sequelize.DataTypes.DECIMAL,
            durationUnit: _sequelize.DataTypes.STRING,
            indication: _sequelize.DataTypes.STRING,
            isPhoneOrder: _sequelize.DataTypes.BOOLEAN,
            notes: _sequelize.DataTypes.STRING,
            pharmacyNotes: _sequelize.DataTypes.STRING,
            displayPharmacyNotesInMar: _sequelize.DataTypes.BOOLEAN,
            quantity: _sequelize.DataTypes.INTEGER,
            discontinued: _sequelize.DataTypes.BOOLEAN,
            discontinuedDate: _sequelize.DataTypes.STRING,
            discontinuingReason: _sequelize.DataTypes.STRING,
            repeats: _sequelize.DataTypes.INTEGER
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL,
            hooks: {
                afterCreate: async (prescription)=>{
                    if (prescription.durationValue && prescription.durationUnit) {
                        const { add } = await Promise.resolve().then(()=>/*#__PURE__*/ _interop_require_wildcard(require("date-fns")));
                        prescription.endDate = add(new Date(prescription.startDate), {
                            [prescription.durationUnit]: prescription.durationValue
                        }).toISOString();
                    }
                    await prescription.save();
                },
                afterUpdate: async (prescription, options)=>{
                    if (prescription.changed('pharmacyNotes')) {
                        await models.Notification.pushNotification(_constants.NOTIFICATION_TYPES.PHARMACY_NOTE, prescription.dataValues, {
                            transaction: options.transaction
                        });
                    }
                    if (prescription.changed('discontinued') && prescription.discontinued) {
                        await models.MedicationAdministrationRecord.removeInvalidMedicationAdministrationRecords(options.transaction);
                    }
                }
            }
        });
    }
    static initRelations(models) {
        this.belongsTo(models.User, {
            foreignKey: 'prescriberId',
            as: 'prescriber'
        });
        this.belongsTo(models.User, {
            foreignKey: 'discontinuingClinicianId',
            as: 'discontinuingClinician'
        });
        this.hasOne(models.EncounterPrescription, {
            foreignKey: 'prescriptionId',
            as: 'encounterPrescription'
        });
        this.belongsToMany(models.Encounter, {
            through: models.EncounterPrescription,
            foreignKey: 'prescriptionId',
            as: 'encounters'
        });
        this.hasOne(models.PatientOngoingPrescription, {
            foreignKey: 'prescriptionId',
            as: 'patientOngoingPrescription'
        });
        this.belongsToMany(models.Patient, {
            through: models.PatientOngoingPrescription,
            foreignKey: 'prescriptionId',
            as: 'patients'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'medicationId',
            as: 'medication'
        });
        this.hasMany(models.MedicationAdministrationRecord, {
            foreignKey: 'prescriptionId',
            as: 'medicationAdministrationRecords'
        });
    }
    static getListReferenceAssociations() {
        return [
            'medication',
            'prescriber',
            'discontinuingClinician'
        ];
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return `
      LEFT JOIN encounter_prescriptions ON prescriptions.id = encounter_prescriptions.prescription_id
      LEFT JOIN encounters ON encounter_prescriptions.encounter_id = encounters.id
      LEFT JOIN patient_ongoing_prescriptions ON prescriptions.id = patient_ongoing_prescriptions.prescription_id
      WHERE (
        (encounters.patient_id IS NOT NULL AND encounters.patient_id IN (SELECT patient_id FROM ${markedForSyncPatientsTable}))
        OR 
        (patient_ongoing_prescriptions.patient_id IS NOT NULL AND patient_ongoing_prescriptions.patient_id IN (SELECT patient_id FROM ${markedForSyncPatientsTable}))
      )
      AND prescriptions.updated_at_sync_tick > :since
    `;
    }
    static buildSyncLookupQueryDetails() {
        return {
            select: (0, _buildEncounterLinkedLookupFilter.buildEncounterLinkedLookupSelect)(this, {
                patientId: 'COALESCE(encounters.patient_id, patient_ongoing_prescriptions.patient_id)'
            }),
            joins: `
        LEFT JOIN encounter_prescriptions ON prescriptions.id = encounter_prescriptions.prescription_id
        LEFT JOIN encounters ON encounter_prescriptions.encounter_id = encounters.id
        LEFT JOIN patient_ongoing_prescriptions ON prescriptions.id = patient_ongoing_prescriptions.prescription_id
        LEFT JOIN locations ON encounters.location_id = locations.id
        LEFT JOIN facilities ON locations.facility_id = facilities.id
      `
        };
    }
};

//# sourceMappingURL=Prescription.js.map