"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "Notification", {
    enumerable: true,
    get: function() {
        return Notification;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _dateTime = require("@tamanu/utils/dateTime");
const _logging = require("@tamanu/shared/services/logging");
const _Model = require("./Model");
const _model = require("../types/model");
const _sync = require("../sync");
function _define_property(obj, key, value) {
    if (key in obj) {
        Object.defineProperty(obj, key, {
            value: value,
            enumerable: true,
            configurable: true,
            writable: true
        });
    } else {
        obj[key] = value;
    }
    return obj;
}
const NOTIFICATION_TYPE_VALUES = Object.values(_constants.NOTIFICATION_TYPES);
const NOTIFICATION_STATUS_VALUES = Object.values(_constants.NOTIFICATION_STATUSES);
let Notification = class Notification extends _Model.Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            type: {
                type: _sequelize.DataTypes.ENUM(...NOTIFICATION_TYPE_VALUES),
                allowNull: false
            },
            status: {
                type: _sequelize.DataTypes.ENUM(...NOTIFICATION_STATUS_VALUES),
                defaultValue: _constants.NOTIFICATION_STATUSES.UNREAD,
                allowNull: false
            },
            createdTime: (0, _model.dateTimeType)('createdTime', {
                allowNull: true
            }),
            metadata: {
                type: _sequelize.DataTypes.JSONB,
                allowNull: false,
                defaultValue: {}
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL
        });
    }
    static initRelations(models) {
        this.belongsTo(models.User, {
            foreignKey: 'userId',
            as: 'user'
        });
        this.belongsTo(models.Patient, {
            foreignKey: 'patientId',
            as: 'patient'
        });
    }
    static buildSyncLookupQueryDetails() {
        return (0, _sync.buildPatientLinkedLookupFilter)(this);
    }
    static getFullReferenceAssociations() {
        const { models } = this.sequelize;
        return [
            {
                model: models.User,
                as: 'user',
                attributes: [
                    'id',
                    'displayName'
                ]
            },
            {
                model: models.Patient,
                as: 'patient',
                attributes: [
                    'id',
                    'firstName',
                    'middleName',
                    'lastName',
                    'displayId'
                ]
            }
        ];
    }
    static async pushNotification(type, metadata, options) {
        try {
            const additionalMetadata = {};
            const { models } = this.sequelize;
            let patientId;
            let userId;
            switch(type){
                case _constants.NOTIFICATION_TYPES.IMAGING_REQUEST:
                    {
                        userId = metadata.requestedById;
                        const encounter = await models.Encounter.findByPk(metadata.encounterId);
                        patientId = encounter.patientId;
                        break;
                    }
                case _constants.NOTIFICATION_TYPES.LAB_REQUEST:
                    {
                        userId = metadata.requestedById;
                        const encounter = await models.Encounter.findByPk(metadata.encounterId);
                        patientId = encounter.patientId;
                        break;
                    }
                case _constants.NOTIFICATION_TYPES.PHARMACY_NOTE:
                    {
                        userId = metadata.prescriberId;
                        const encounterPrescription = await models.EncounterPrescription.findOne({
                            where: {
                                prescriptionId: metadata.id
                            },
                            include: [
                                'encounter'
                            ]
                        });
                        patientId = encounterPrescription.encounter.patientId;
                        additionalMetadata.encounterId = encounterPrescription.encounterId;
                        break;
                    }
                default:
                    return;
            }
            if (!patientId || !userId) {
                return;
            }
            await this.create({
                type,
                metadata: {
                    ...metadata,
                    ...additionalMetadata
                },
                userId,
                patientId,
                createdTime: (0, _dateTime.getCurrentDateTimeString)()
            }, options);
        } catch (error) {
            _logging.log.error('Error pushing notification', error);
        }
    }
};
_define_property(Notification, "buildPatientSyncFilter", _sync.buildPatientSyncFilterViaPatientId);

//# sourceMappingURL=Notification.js.map