"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "SurveyResponse", {
    enumerable: true,
    get: function() {
        return SurveyResponse;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _errors = require("@tamanu/shared/errors");
const _calculations = require("@tamanu/shared/utils/calculations");
const _fields = require("@tamanu/shared/utils/fields");
const _getPatientDataDbLocation = require("@tamanu/shared/utils/getPatientDataDbLocation");
const _dateTime = require("@tamanu/utils/dateTime");
const _Model = require("./Model");
const _buildEncounterLinkedSyncFilter = require("../sync/buildEncounterLinkedSyncFilter");
const _model = require("../types/model");
const _buildEncounterLinkedLookupFilter = require("../sync/buildEncounterLinkedLookupFilter");
async function createPatientIssues(models, questions, patientId) {
    const issueQuestions = questions.filter((q)=>q.dataElement.type === _constants.PROGRAM_DATA_ELEMENT_TYPES.PATIENT_ISSUE);
    for (const question of issueQuestions){
        const { config: configString } = question;
        const config = JSON.parse(configString) || {};
        if (!config.issueNote || !config.issueType) {
            throw new _errors.InvalidOperationError(`Ill-configured PatientIssue with config: ${configString}`);
        }
        await models.PatientIssue.create({
            patientId,
            type: config.issueType,
            note: config.issueNote
        });
    }
}
/** Returns in the format:
 * {
 *  Patient: { key1: 'value1' },
 *  PatientAdditionalData: { key1: 'value1' },
 * }
 */ const getFieldsToWrite = async (models, questions, answers)=>{
    const recordValuesByModel = {};
    const patientDataQuestions = questions.filter((q)=>q.dataElement.type === _constants.PROGRAM_DATA_ELEMENT_TYPES.PATIENT_DATA);
    for (const question of patientDataQuestions){
        const { dataElement, config: configString } = question;
        const config = JSON.parse(configString) || {};
        if (!config.writeToPatient) {
            continue;
        }
        const { fieldName: configFieldName } = config.writeToPatient || {};
        if (!configFieldName) {
            throw new Error('No fieldName defined for writeToPatient config');
        }
        const value = answers[dataElement.id];
        const { modelName, fieldName } = await (0, _getPatientDataDbLocation.getPatientDataDbLocation)(configFieldName, models);
        if (!modelName) {
            throw new Error(`Unknown fieldName: ${configFieldName}`);
        }
        if (!recordValuesByModel[modelName]) recordValuesByModel[modelName] = {};
        recordValuesByModel[modelName][fieldName] = value;
    }
    return recordValuesByModel;
};
/**
 * DUPLICATED IN mobile/App/models/SurveyResponse.ts
 * Please keep in sync
 */ async function writeToPatientFields(models, facilityId, questions, answers, patientId, surveyId, userId, submittedTime) {
    const valuesByModel = await getFieldsToWrite(models, questions, answers);
    if (valuesByModel.Patient) {
        const patient = await models.Patient.findByPk(patientId);
        await patient?.update(valuesByModel.Patient);
    }
    if (valuesByModel.PatientFieldValue) {
        const patient = await models.Patient.findByPk(patientId);
        await patient?.writeFieldValues(valuesByModel.PatientFieldValue);
    }
    if (valuesByModel.PatientAdditionalData) {
        const pad = await models.PatientAdditionalData.getOrCreateForPatient(patientId);
        await pad.update(valuesByModel.PatientAdditionalData);
    }
    if (valuesByModel.PatientProgramRegistration) {
        const survey = await models.Survey.findByPk(surveyId);
        const programRegistryDetail = await models.ProgramRegistry.findOne({
            where: {
                programId: survey?.programId,
                visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT
            }
        });
        if (!programRegistryDetail?.id) {
            throw new Error('No program registry configured for the current form');
        }
        // Check if a record already exists with the given patientId and programRegistryId
        const existingRegistration = await models.PatientProgramRegistration.findOne({
            where: {
                patientId,
                programRegistryId: programRegistryDetail.id
            }
        });
        const registrationData = {
            patientId,
            programRegistryId: programRegistryDetail.id,
            date: submittedTime,
            ...valuesByModel.PatientProgramRegistration,
            registeringFacilityId: valuesByModel.PatientProgramRegistration.registeringFacilityId || facilityId,
            clinicianId: valuesByModel.PatientProgramRegistration.clinicianId || userId
        };
        if (existingRegistration) {
            // Update the existing record
            await existingRegistration.update(registrationData);
        } else {
            // Create a new record
            await models.PatientProgramRegistration.create(registrationData);
        }
    }
}
async function handleSurveyResponseActions(models, facilityId, questions, answers, patientId, surveyId, userId, submittedTime) {
    const activeQuestions = (0, _fields.getActiveActionComponents)(questions, answers);
    await createPatientIssues(models, activeQuestions, patientId);
    await writeToPatientFields(models, facilityId, activeQuestions, answers, patientId, surveyId, userId, submittedTime);
}
let SurveyResponse = class SurveyResponse extends _Model.Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            startTime: (0, _model.dateTimeType)('startTime', {
                allowNull: true
            }),
            endTime: (0, _model.dateTimeType)('endTime', {
                allowNull: true
            }),
            result: {
                type: _sequelize.DataTypes.FLOAT,
                allowNull: true
            },
            resultText: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: true
            },
            notified: {
                type: _sequelize.DataTypes.BOOLEAN,
                allowNull: true
            },
            metadata: {
                type: _sequelize.DataTypes.JSONB,
                allowNull: true
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL
        });
    }
    static initRelations(models) {
        this.belongsTo(models.User, {
            foreignKey: 'userId',
            as: 'user'
        });
        this.belongsTo(models.Survey, {
            foreignKey: 'surveyId',
            as: 'survey'
        });
        this.belongsTo(models.Encounter, {
            foreignKey: 'encounterId',
            as: 'encounter'
        });
        this.hasMany(models.SurveyResponseAnswer, {
            foreignKey: 'responseId',
            as: 'answers'
        });
        this.hasOne(models.Referral, {
            foreignKey: 'surveyResponseId',
            as: 'referral'
        });
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilter)([
            this.tableName,
            'encounters'
        ], markedForSyncPatientsTable);
    }
    static buildSyncLookupQueryDetails() {
        return (0, _buildEncounterLinkedLookupFilter.buildEncounterLinkedLookupFilter)(this);
    }
    static async getSurveyEncounter({ encounterId, patientId, forceNewEncounter, reasonForEncounter, ...responseData }) {
        if (!this.sequelize.isInsideTransaction()) {
            throw new Error('SurveyResponse.getSurveyEncounter must always run inside a transaction!');
        }
        const { Encounter } = this.sequelize.models;
        if (encounterId) {
            return Encounter.findByPk(encounterId);
        }
        if (!patientId) {
            throw new _errors.InvalidOperationError('A survey response must have an encounter or patient ID attached');
        }
        if (!forceNewEncounter) {
            // find open encounter
            const openEncounter = await Encounter.findOne({
                where: {
                    patientId,
                    endDate: null
                }
            });
            if (openEncounter) {
                return openEncounter;
            }
        }
        const { departmentId, examinerId, userId, locationId } = responseData;
        // need to create a new encounter with examiner set as the user who submitted the survey.
        const newEncounter = await Encounter.create({
            patientId,
            encounterType: 'surveyResponse',
            reasonForEncounter,
            departmentId,
            examinerId: examinerId || userId,
            locationId,
            // Survey responses will usually have a startTime and endTime and we prefer to use that
            // for the encounter to ensure the times are set in the browser timezone
            startDate: responseData.startTime ? responseData.startTime : (0, _dateTime.getCurrentDateTimeString)(),
            actorId: userId
        });
        return newEncounter.update({
            endDate: responseData.endTime ? responseData.endTime : (0, _dateTime.getCurrentDateTimeString)(),
            systemNote: 'Automatically discharged',
            discharge: {
                note: 'Automatically discharged after survey completion'
            }
        });
    }
    static async createWithAnswers(data) {
        if (!this.sequelize.isInsideTransaction()) {
            throw new Error('SurveyResponse.createWithAnswers must always run inside a transaction!');
        }
        const { models } = this.sequelize;
        const { answers, surveyId, patientId, encounterId, forceNewEncounter, facilityId, ...responseData } = data;
        // ensure survey exists
        const survey = await models.Survey.findByPk(surveyId);
        if (!survey) {
            throw new _errors.InvalidOperationError(`Invalid survey ID: ${surveyId}`);
        }
        // figure out if its a vital survey response
        const vitalsSurvey = await models.Survey.getVitalsSurvey();
        // use optional chaining because vitals survey might not exist
        const isVitalSurvey = surveyId === vitalsSurvey?.id;
        const questions = await models.SurveyScreenComponent.getComponentsForSurvey(surveyId);
        const calculatedAnswers = (0, _calculations.runCalculations)(questions, answers);
        const finalAnswers = {
            ...answers,
            ...calculatedAnswers
        };
        const encounter = await this.getSurveyEncounter({
            encounterId,
            patientId,
            forceNewEncounter,
            reasonForEncounter: 'Form response',
            ...responseData
        });
        const { result, resultText } = (0, _fields.getResultValue)(questions, answers, {
            encounterType: encounter.encounterType
        });
        const record = await SurveyResponse.create({
            patientId,
            surveyId,
            encounterId: encounter.id,
            result,
            resultText,
            // put responseData last to allow for user to override
            // resultText by including it in the data
            // this is used by reports test where the resultText
            // is included in the payload
            ...responseData,
            notified: responseData?.endTime && survey?.notifiable ? false : null
        });
        const findDataElement = (id)=>{
            const component = questions.find((c)=>c.dataElement.id === id);
            if (!component) return null;
            return component.dataElement;
        };
        // create answer records
        for (const a of Object.entries(finalAnswers)){
            const [dataElementId, value] = a;
            const dataElement = findDataElement(dataElementId);
            if (!dataElement) {
                throw new Error(`no data element for question: ${dataElementId}`);
            }
            const body = (0, _fields.getStringValue)(dataElement.type, value);
            // Don't create null answers
            if (body === null) {
                continue;
            }
            const answer = await models.SurveyResponseAnswer.create({
                dataElementId: dataElement.id,
                body,
                responseId: record.id
            });
            if (!isVitalSurvey || body === '') continue;
            // Generate initial vital log
            await models.VitalLog.create({
                date: record.endTime || (0, _dateTime.getCurrentDateTimeString)(),
                newValue: body,
                recordedById: responseData.userId,
                answerId: answer.id
            });
        }
        await handleSurveyResponseActions(models, facilityId, questions, finalAnswers, encounter.patientId, surveyId, responseData.userId, responseData.endTime);
        return record;
    }
};

//# sourceMappingURL=SurveyResponse.js.map