"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    dataGenerator: function() {
        return dataGenerator;
    },
    queryCovidVaccineListData: function() {
        return queryCovidVaccineListData;
    },
    reportColumnTemplate: function() {
        return reportColumnTemplate;
    }
});
const _sequelize = require("sequelize");
const _datefns = require("date-fns");
const _utilities = require("./utilities");
const _dateTime = require("@tamanu/utils/dateTime");
const reportColumnTemplate = [
    {
        title: 'Patient Name',
        accessor: (data)=>data.patientName
    },
    {
        title: 'UID',
        accessor: (data)=>data.uid
    },
    {
        title: 'DOB',
        accessor: (data)=>data.dob
    },
    {
        title: 'Sex',
        accessor: (data)=>data.sex
    },
    {
        title: 'Village',
        accessor: (data)=>data.village
    },
    {
        title: 'Vaccine name',
        accessor: (data)=>data.vaccineName
    },
    {
        title: 'Vaccine status',
        accessor: (data)=>data.vaccineStatus
    },
    {
        title: 'Schedule',
        accessor: (data)=>data.doseLabel
    },
    {
        title: 'Vaccine date',
        accessor: (data)=>data.vaccineDate
    },
    {
        title: 'Batch',
        accessor: (data)=>data.batch
    },
    {
        title: 'Vaccinator',
        accessor: (data)=>data.vaccinator
    }
];
function parametersToSqlWhere(parameters) {
    const newParameters = {
        ...parameters,
        fromDate: (0, _dateTime.toDateTimeString)((0, _datefns.startOfDay)(parameters.fromDate ? (0, _datefns.parseISO)(parameters.fromDate) : (0, _datefns.subDays)(new Date(), 30)))
    };
    if (parameters.toDate) {
        newParameters.toDate = (0, _dateTime.toDateTimeString)((0, _datefns.endOfDay)((0, _datefns.parseISO)(parameters.toDate)));
    }
    const whereClause = Object.entries(newParameters).filter(([, val])=>val).reduce((where, [key, value])=>{
        const newWhere = {
            ...where
        };
        switch(key){
            case 'village':
                newWhere['$encounter->patient.village_id$'] = value;
                break;
            case 'fromDate':
                if (!newWhere.date) {
                    newWhere.date = {};
                }
                newWhere.date[_sequelize.Op.gte] = value;
                break;
            case 'toDate':
                if (!newWhere.date) {
                    newWhere.date = {};
                }
                newWhere.date[_sequelize.Op.lte] = value;
                break;
            case 'category':
                newWhere['$scheduledVaccine.category$'] = value;
                break;
            case 'vaccine':
                newWhere['$scheduledVaccine.label$'] = value;
                break;
            default:
                break;
        }
        return newWhere;
    }, {});
    return whereClause;
}
async function queryCovidVaccineListData(models, parameters) {
    const result = await models.AdministeredVaccine.findAll({
        include: [
            {
                model: models.Encounter,
                as: 'encounter',
                include: [
                    {
                        model: models.Patient,
                        as: 'patient',
                        include: [
                            {
                                model: models.ReferenceData,
                                as: 'village'
                            }
                        ]
                    },
                    {
                        model: models.User,
                        as: 'examiner'
                    }
                ]
            },
            {
                model: models.ScheduledVaccine,
                as: 'scheduledVaccine'
            },
            {
                model: models.User,
                as: 'recorder'
            }
        ],
        where: parametersToSqlWhere(parameters),
        order: [
            [
                {
                    model: models.Encounter,
                    as: 'encounter'
                },
                {
                    model: models.Patient,
                    as: 'patient'
                },
                'id',
                'ASC'
            ],
            [
                'date',
                'ASC'
            ]
        ]
    });
    const administeredVaccines = result.map((r)=>r.get({
            plain: true
        }));
    const reportData = [];
    for (const vaccine of administeredVaccines){
        if (!vaccine.encounter?.patientId) {
            continue;
        }
        const { encounter: { patient: { id: patientId, displayId, firstName, lastName, dateOfBirth, village, sex }, examiner: { displayName: examinerName } }, date, status, batch, scheduledVaccine: { doseLabel, label: vaccineName }, recorder } = vaccine;
        const vaccinator = vaccine.givenBy ?? recorder?.displayName ?? examinerName;
        const record = {
            patientId,
            patientName: `${firstName} ${lastName}`,
            uid: displayId,
            dob: (0, _dateTime.format)(dateOfBirth, 'dd-MM-yyyy'),
            sex,
            village: village?.name,
            vaccineName,
            doseLabel,
            vaccineStatus: status === 'GIVEN' ? 'Yes' : 'No',
            vaccineDate: (0, _dateTime.format)(date, 'dd-MM-yyyy'),
            batch: status === 'GIVEN' ? batch : '',
            vaccinator: status === 'GIVEN' ? vaccinator : ''
        };
        reportData.push(record);
    }
    return reportData;
}
async function dataGenerator({ models }, parameters) {
    const queryResults = await queryCovidVaccineListData(models, parameters);
    return (0, _utilities.generateReportFromQueryData)(queryResults, reportColumnTemplate);
}

//# sourceMappingURL=vaccine-list.js.map