"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "dataGenerator", {
    enumerable: true,
    get: function() {
        return dataGenerator;
    }
});
const _lodash = require("lodash");
const _sequelize = require("sequelize");
const _datefns = require("date-fns");
const _constants = require("@tamanu/constants");
const _utilities = require("../utilities");
const _utils = require("./utils");
const _dateTime = require("@tamanu/utils/dateTime");
const _constants1 = require("./constants");
const parametersToReferralSqlWhere = (parameters)=>{
    const where = {
        '$surveyResponse.survey_id$': {
            [_sequelize.Op.in]: _constants1.REFERRAL_SURVEY_IDS
        },
        status: [
            _constants.REFERRAL_STATUSES.PENDING
        ]
    };
    if (!parameters || !Object.keys(parameters).length) {
        return where;
    }
    if (parameters.fromDate || parameters.toDate) {
        where['$surveyResponse.end_time$'] = {};
    }
    if (parameters.fromDate) {
        where['$surveyResponse.end_time$'][_sequelize.Op.gte] = (0, _dateTime.toDateTimeString)((0, _datefns.startOfDay)((0, _datefns.parseISO)(parameters.fromDate)));
    }
    if (parameters.toDate) {
        where['$surveyResponse.end_time$'][_sequelize.Op.lte] = (0, _dateTime.toDateTimeString)((0, _datefns.endOfDay)((0, _datefns.parseISO)(parameters.toDate)));
    }
    if (parameters.surveyId) {
        delete where['$surveyResponse.survey_id$'][_sequelize.Op.in];
        where['$surveyResponse.survey_id$'][_sequelize.Op.eq] = parameters.surveyId;
    }
    return where;
};
const getPendingReferrals = async (models, surveyIds, parameters)=>models.Referral.findAll({
        where: parametersToReferralSqlWhere(parameters),
        include: [
            {
                model: models.Encounter,
                as: 'initiatingEncounter'
            },
            {
                model: models.SurveyResponse,
                as: 'surveyResponse',
                where: {
                    surveyId: {
                        [_sequelize.Op.in]: surveyIds
                    }
                }
            }
        ],
        order: [
            [
                {
                    model: models.SurveyResponse,
                    as: 'surveyResponse'
                },
                'end_time',
                'ASC'
            ]
        ]
    });
const sortReferrals = (r1, r2)=>{
    const patientId1 = r1.initiatingEncounter?.patientId;
    const patientId2 = r2.initiatingEncounter?.patientId;
    const surveyGroupKey1 = (0, _utils.getSurveyGroupKey)(r1.surveyResponse?.surveyId);
    const surveyGroupKey2 = (0, _utils.getSurveyGroupKey)(r2.surveyResponse?.surveyId);
    const responseTime1 = r1.surveyResponse?.endTime;
    const responseTime2 = r2.surveyResponse?.endTime;
    return patientId1.localeCompare(patientId2) || surveyGroupKey1.localeCompare(surveyGroupKey2) || (0, _datefns.isAfter)(responseTime1, responseTime2);
};
const dataGenerator = async ({ models }, parameters = {})=>{
    const referrals = await getPendingReferrals(models, _constants1.REFERRAL_SURVEY_IDS, parameters);
    const filteredReferrals = await (0, _utils.removeDuplicatedReferralsPerDate)(referrals);
    const sortedReferrals = filteredReferrals.sort(sortReferrals);
    const answerWhereClause = (0, _utils.parametersToAnswerSqlWhere)(parameters);
    const rawAnswers = await (0, _utilities.getAnswers)(models, answerWhereClause);
    const filteredAnswers = await (0, _utils.transformAndRemoveDuplicatedAnswersPerDate)(models, rawAnswers, _constants1.ALL_SURVEY_IDS);
    const patientById = await (0, _utils.getPatientById)(models, rawAnswers);
    const answersByPatientSurveyDataElement = (0, _lodash.keyBy)(filteredAnswers, (a)=>{
        const responseDate = (0, _dateTime.format)(a.responseEndTime, 'dd-MM-yyyy');
        const surveyGroupKey = (0, _utils.getSurveyGroupKey)(a.surveyId);
        return (0, _utils.getPerPatientPerSurveyPerDatePerElementKey)(a.patientId, surveyGroupKey, responseDate, a.dataElementId);
    });
    const reportData = [];
    // Report should create a new line for each patient with the below referrals in a status of pending:
    // CVD Primary Screening Referral
    // Breast Cancer Primary Screening Referral
    // Cervical Cancer Primary Screening Referral
    // If there are multiple referral submission on the same date (with same referral survey), it should take the
    // latest answer for each data element regardless of which referral submission
    // For each referral, pull most recent corresponding screening details (on the same date date)
    for (const referral of sortedReferrals){
        const referralSurveyResponse = referral.surveyResponse;
        const { patientId } = referral.initiatingEncounter;
        const patient = patientById[patientId];
        const patientAdditionalData = patient.additionalData?.[0];
        const referralDate = (0, _dateTime.format)(referralSurveyResponse.endTime, 'dd-MM-yyyy');
        const { surveyId } = referralSurveyResponse;
        const surveyGroupKey = (0, _utils.getSurveyGroupKey)(surveyId);
        const age = patient.dateOfBirth ? (0, _dateTime.ageInYears)(patient.dateOfBirth) : '';
        const recordData = {
            firstName: patient.firstName,
            lastName: patient.lastName,
            displayId: patient.displayId,
            age,
            gender: patient.sex,
            ethnicity: patientAdditionalData?.ethnicity?.name,
            contactNumber: patientAdditionalData?.primaryContactNumber,
            village: patient.village?.name,
            medicalArea: patientAdditionalData?.medicalArea?.name,
            nursingZone: patientAdditionalData?.nursingZone?.name,
            referralCreated: _constants1.REFERRAL_NAME_BY_SURVEY_GROUP_KEY[surveyGroupKey]
        };
        const referralDataElements = (0, _utils.getReferralDataElements)(surveyGroupKey);
        Object.entries(referralDataElements).forEach(([dataKey, dataElementId])=>{
            recordData[dataKey] = (0, _utils.getCachedAnswer)(answersByPatientSurveyDataElement, patientId, surveyGroupKey, referralDate, dataElementId);
        });
        const formDataElements = (0, _utils.getFormDataElements)(surveyGroupKey);
        Object.entries(formDataElements).forEach(([dataKey, dataElementId])=>{
            recordData[dataKey] = (0, _utils.getCachedAnswer)(answersByPatientSurveyDataElement, patientId, surveyGroupKey, referralDate, dataElementId);
        });
        reportData.push(recordData);
    }
    const sortedReportData = reportData.sort(({ dateOfReferral: date1 }, { dateOfReferral: date2 })=>{
        if (date2 && !date1) return 1;
        if (date1 && !date2) return -1;
        if (!date1 && !date2) return 0;
        // Sort oldest to most recent
        return (0, _datefns.parseISO)(date1) - (0, _datefns.parseISO)(date2);
    });
    return (0, _utilities.generateReportFromQueryData)(sortedReportData, _constants1.PRIMARY_SCREENING_PENDING_REFERRALS_REPORT_COLUMN_TEMPLATE);
};

//# sourceMappingURL=fiji-ncd-primary-screening-pending-referrals-line-list.js.map