"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "CertifiableVaccine", {
    enumerable: true,
    get: function() {
        return CertifiableVaccine;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _errors = require("@tamanu/shared/errors");
const _Model = require("./Model");
let CertifiableVaccine = class CertifiableVaccine extends _Model.Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            // ICD11 code for the vaccine type
            icd11DrugCode: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false
            },
            // ICD11 code for the targeted disease
            icd11DiseaseCode: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false
            },
            // SNOMED CT or ATC code for the vaccine type
            vaccineCode: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false
            },
            // SNOMED CT or ATC code for targeted disease
            targetCode: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            },
            // EU authorisation code for the vaccine product
            euProductCode: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            },
            maximumDosage: {
                type: _sequelize.DataTypes.INTEGER,
                allowNull: false,
                defaultValue: 1
            }
        }, {
            ...options,
            // This is essentially reference/imported data
            syncDirection: _constants.SYNC_DIRECTIONS.PULL_FROM_CENTRAL,
            validate: {
                mustHaveVaccine () {
                    if (!this.deletedAt && !this.vaccineId) {
                        throw new _errors.InvalidOperationError('A certifiable vaccine must have a vaccine.');
                    }
                }
            }
        });
    }
    static getListReferenceAssociations() {
        return [
            'encounter',
            'scheduledVaccine'
        ];
    }
    static initRelations(models) {
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'vaccineId',
            as: 'vaccine'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'manufacturerId',
            as: 'manufacturer'
        });
    }
    static async allVaccineIds(euDccOnly = false) {
        let all = await CertifiableVaccine.findAll();
        if (euDccOnly) {
            all = all.filter((v)=>v.usableForEuDcc());
        }
        return all.map((vc)=>vc.vaccineId);
    }
    usableForEuDcc() {
        return this.euProductCode !== null && this.targetCode !== null && this.manufacturerId !== null;
    }
    static buildSyncFilter() {
        return null; // syncs everywhere
    }
    static async buildSyncLookupQueryDetails() {
        return null; // syncs everywhere
    }
};

//# sourceMappingURL=CertifiableVaccine.js.map