"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "pullIncomingChanges", {
    enumerable: true,
    get: function() {
        return pullIncomingChanges;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _lodash = require("lodash");
const _logging = require("@tamanu/shared/services/logging");
const _sync = require("@tamanu/database/sync");
const _sleepAsync = require("@tamanu/utils/sleepAsync");
const _calculatePageLimit = require("./calculatePageLimit");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const { persistedCacheBatchSize, pauseBetweenCacheBatchInMilliseconds } = _config.default.sync;
const pullIncomingChanges = async (centralServer, sequelize, sessionId, since)=>{
    // initiating pull also returns the sync tick (or point on the sync timeline) that the
    // central server considers this session will be up to after pulling all changes
    _logging.log.info('FacilitySyncManager.pull.waitingForCentral');
    const { totalToPull, pullUntil } = await centralServer.initiatePull(sessionId, since);
    _logging.log.info('FacilitySyncManager.pulling', {
        since,
        totalToPull
    });
    let fromId;
    let limit = (0, _calculatePageLimit.calculatePageLimit)();
    let totalPulled = 0;
    // pull changes a page at a time
    while(totalPulled < totalToPull){
        _logging.log.debug('FacilitySyncManager.pull.pullingPage', {
            fromId,
            limit
        });
        const startTime = Date.now();
        const records = await centralServer.pull(sessionId, {
            fromId,
            limit
        });
        fromId = records[records.length - 1]?.id;
        totalPulled += records.length;
        const pullTime = Date.now() - startTime;
        if (!records.length) {
            _logging.log.debug(`FacilitySyncManager.pull.noMoreChanges`);
            break;
        }
        _logging.log.info('FacilitySyncManager.savingChangesToSnapshot', {
            count: records.length
        });
        const recordsToSave = records.map((r)=>({
                ...r,
                data: {
                    ...r.data,
                    updatedAtSyncTick: _sync.SYNC_TICK_FLAGS.INCOMING_FROM_CENTRAL_SERVER
                },
                direction: _sync.SYNC_SESSION_DIRECTION.INCOMING
            }));
        // This is an attempt to avoid storing all the pulled data
        // in the memory because we might run into memory issue when:
        // 1. During the first sync when there is a lot of data to load
        // 2. When a huge number of data is imported to sync and the facility syncs it down
        // So store the data in a sync snapshot table instead and will persist it to the actual tables later
        for (const batchOfRows of (0, _lodash.chunk)(recordsToSave, persistedCacheBatchSize)){
            await (0, _sync.insertSnapshotRecords)(sequelize, sessionId, batchOfRows);
            await (0, _sleepAsync.sleepAsync)(pauseBetweenCacheBatchInMilliseconds);
        }
        limit = (0, _calculatePageLimit.calculatePageLimit)(limit, pullTime);
    }
    return {
        totalPulled: totalToPull,
        pullUntil
    };
};

//# sourceMappingURL=pullIncomingChanges.js.map