"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "generateIPSBundle", {
    enumerable: true,
    get: function() {
        return generateIPSBundle;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _uuid = require("uuid");
const _constants = require("@tamanu/constants");
const _fhir = require("@tamanu/shared/utils/fhir");
const _helpers = require("./helpers");
const _utils = require("./utils");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const generateIPSBundle = async (fhirPatientId, user, models)=>{
    const dataDictionariesIps = _config.default.hl7.dataDictionaries.ips;
    const integrationsIps = _config.default.integrations.ips;
    const fhirPatient = await models.FhirPatient.findByPk(fhirPatientId);
    if (!fhirPatient) throw new _fhir.NotFound(`No FHIR patient with id ${fhirPatientId}`);
    const patientId = fhirPatient.upstreamId;
    if (!patientId) throw new _fhir.NotFound(`No upstream patient for fhir patient with id ${fhirPatientId}`);
    const patient = await models.Patient.findByPk(patientId);
    if (!patient) throw new _fhir.NotFound(`No public patient with id ${patientId}`);
    // We set this to an ID independent of the DB ecosystem
    // Alternatively, we could fetch the patient from the fhir schema in the DB
    patient.id = fhirPatientId;
    patient.displayName = (0, _utils.getPatientDisplayName)(patient);
    const [medicationStatements, allergyIntolerances, conditions, immunizations, hl7Patient] = await Promise.all([
        (0, _helpers.getMedicationStatements)({
            patient,
            models,
            dataDictionariesIps
        }),
        (0, _helpers.getAllergyIntolerances)({
            patient,
            models,
            dataDictionariesIps
        }),
        (0, _helpers.getConditions)({
            patient,
            models,
            dataDictionariesIps
        }),
        (0, _helpers.getImmunizations)({
            patient,
            models,
            dataDictionariesIps
        }),
        (0, _helpers.getHl7Patient)({
            patient,
            models
        })
    ]);
    const now = new Date();
    const composition = (0, _helpers.getComposition)({
        patient,
        user,
        integrationsIps,
        now,
        medicationStatements,
        allergyIntolerances,
        conditions,
        immunizations
    });
    const bundle = {
        id: (0, _uuid.v4)(),
        resourceType: _constants.FHIR_RESOURCE_TYPES.BUNDLE,
        language: 'en-AU',
        identifier: {
            system: 'urn:oid:2.16.724.4.8.10.200.10',
            value: (0, _uuid.v4)()
        },
        type: 'document',
        timestamp: (0, _fhir.formatFhirDate)(now),
        entry: [
            (0, _utils.getBundleEntryFromResource)(composition),
            (0, _utils.getBundleEntryFromResource)(hl7Patient),
            ...conditions.map((condition)=>(0, _utils.getBundleEntryFromResource)(condition)),
            ...medicationStatements.map((medicationStatement)=>(0, _utils.getBundleEntryFromResource)(medicationStatement)),
            ...allergyIntolerances.map((allergyIntolerance)=>(0, _utils.getBundleEntryFromResource)(allergyIntolerance)),
            ...immunizations.map((immunization)=>(0, _utils.getBundleEntryFromResource)(immunization))
        ]
    };
    return {
        patient,
        bundle
    };
};

//# sourceMappingURL=bundleGenerator.js.map