import React from 'react';
import { Document, View, StyleSheet } from '@react-pdf/renderer';
import { Table } from './Table';
import { styles, Box, Watermark, CertificateHeader, PageBreakPadding, FixedFooter, FixedHeader } from './Layout';
import { PatientDetailsSection } from './PatientDetailsSection';
import { H3 } from './Typography';
import { LetterheadSection } from './LetterheadSection';
import { getDisplayDate } from './getDisplayDate';
import { SigningSection } from './SigningSection';
import { useLanguageContext, withLanguageContext } from '../pdf/languageContext';
import { Page } from '../pdf/Page';
import { Text } from '../pdf/Text';
import { get } from 'lodash';
import { useTextStyles } from './printComponents/MultiPageHeader';
const columns = (getTranslation)=>[
        {
            key: 'date',
            title: getTranslation('pdf.table.column.dateGiven', 'Date given'),
            accessor: ({ date }, getLocalisation)=>date ? getDisplayDate(date, undefined, getLocalisation) : 'Unknown'
        },
        {
            key: 'schedule',
            title: getTranslation('pdf.table.column.schedule', 'Schedule'),
            accessor: ({ scheduledVaccine })=>(scheduledVaccine || {}).doseLabel
        },
        {
            key: 'vaccine',
            title: getTranslation('pdf.table.column.vaccine', 'Vaccine'),
            customStyles: {
                minWidth: 30
            },
            accessor: ({ scheduledVaccine, vaccineName })=>vaccineName || (scheduledVaccine || {}).label
        },
        {
            key: 'vaccineBrand',
            title: getTranslation('pdf.table.column.vaccineBrand', 'Vaccine brand'),
            customStyles: {
                minWidth: 30
            },
            accessor: ({ scheduledVaccine, vaccineBrand })=>vaccineBrand || ((scheduledVaccine || {}).vaccine || {}).name
        },
        {
            key: 'countryName',
            title: getTranslation('pdf.table.column.country', 'Facility/Country'),
            accessor: (record)=>{
                const facility = record.givenElsewhere ? record.givenBy : record.location?.facility?.name;
                return facility || '';
            }
        }
    ];
const vaccineCertificateStyles = StyleSheet.create({
    footerContent: {
        flexDirection: 'row',
        justifyContent: 'space-between',
        alignItems: 'center'
    },
    footerLeft: {
        flexDirection: 'row',
        flexWrap: 'wrap'
    },
    footerRight: {
        flex: 1,
        textAlign: 'right'
    },
    labelText: {
        fontSize: 8,
        color: '#888888'
    },
    valueText: {
        fontSize: 8,
        color: '#888888'
    },
    documentHeaderContent: {
        flexDirection: 'row'
    }
});
const VaccineCertificateHeader = ({ patient })=>{
    const valueStyles = useTextStyles(vaccineCertificateStyles.valueText);
    const labelStyles = useTextStyles(vaccineCertificateStyles.labelText);
    const ValueText = (props)=>/*#__PURE__*/ React.createElement(Text, {
            style: valueStyles,
            ...props
        });
    const LabelText = (props)=>/*#__PURE__*/ React.createElement(Text, {
            bold: true,
            style: labelStyles,
            ...props
        });
    const { getTranslation } = useLanguageContext();
    return /*#__PURE__*/ React.createElement(View, {
        fixed: true,
        render: ({ pageNumber })=>pageNumber > 1 && /*#__PURE__*/ React.createElement(View, {
                style: vaccineCertificateStyles.documentHeaderContent
            }, /*#__PURE__*/ React.createElement(ValueText, null, getTranslation('pdf.vaccineCertificate.immunisationCertificate', 'Immunisation Certificate'), ' ', "|", ' '), /*#__PURE__*/ React.createElement(LabelText, null, getTranslation('pdf.vaccineCertificate.patientName', 'Patient name'), ":", ' '), /*#__PURE__*/ React.createElement(ValueText, null, patient.firstName, " ", patient.lastName, " |", ' '), /*#__PURE__*/ React.createElement(LabelText, null, getTranslation('pdf.vaccineCertificate.patientId', 'Patient ID'), ":", ' '), /*#__PURE__*/ React.createElement(ValueText, null, patient.displayId))
    });
};
const VaccineCertificateComponent = ({ patient, printedBy, printedDate, facilityName, vaccinations, certificateId, watermarkSrc, signingSrc, logoSrc, localisation, getSetting, extraPatientFields, certificateData, healthFacility })=>{
    const { getTranslation } = useLanguageContext();
    const getLocalisation = (key)=>get(localisation, key);
    const countryName = getLocalisation('country.name');
    const data = vaccinations.map((vaccination)=>({
            ...vaccination,
            countryName,
            healthFacility
        }));
    const VaccineCertificateFooter = ()=>/*#__PURE__*/ React.createElement(View, {
            style: vaccineCertificateStyles.footerContent
        }, /*#__PURE__*/ React.createElement(View, {
            style: vaccineCertificateStyles.footerLeft
        }, /*#__PURE__*/ React.createElement(Text, {
            bold: true,
            style: vaccineCertificateStyles.labelText
        }, getTranslation('pdf.vaccineCertificate.printDate', 'Print date'), ":", ' '), /*#__PURE__*/ React.createElement(Text, {
            bold: true,
            style: vaccineCertificateStyles.valueText
        }, getDisplayDate(printedDate), " |", ' '), /*#__PURE__*/ React.createElement(Text, {
            bold: true,
            style: vaccineCertificateStyles.labelText
        }, getTranslation('pdf.vaccineCertificate.printingFacility', 'Printing facility'), ":", ' '), /*#__PURE__*/ React.createElement(Text, {
            bold: true,
            style: vaccineCertificateStyles.valueText
        }, facilityName || healthFacility, " |", ' '), /*#__PURE__*/ React.createElement(Text, {
            bold: true,
            style: vaccineCertificateStyles.labelText
        }, getTranslation('pdf.vaccineCertificate.printedBy', 'Printed by'), ":", ' '), /*#__PURE__*/ React.createElement(Text, {
            bold: true,
            style: vaccineCertificateStyles.valueText
        }, printedBy)), /*#__PURE__*/ React.createElement(View, {
            style: vaccineCertificateStyles.footerRight
        }, /*#__PURE__*/ React.createElement(Text, {
            bold: true,
            style: vaccineCertificateStyles.valueText,
            render: ({ pageNumber, totalPages })=>getTranslation('pdf.pagination', ':currentPage of :totalPages', {
                    replacements: {
                        currentPage: pageNumber,
                        totalPages
                    }
                })
        })));
    return /*#__PURE__*/ React.createElement(Document, null, /*#__PURE__*/ React.createElement(Page, {
        size: "A4",
        style: {
            ...styles.page,
            paddingBottom: 51
        }
    }, /*#__PURE__*/ React.createElement(FixedHeader, null, /*#__PURE__*/ React.createElement(VaccineCertificateHeader, {
        patient: patient
    })), /*#__PURE__*/ React.createElement(View, {
        fixed: true,
        render: ({ pageNumber })=>pageNumber > 1 && /*#__PURE__*/ React.createElement(PageBreakPadding, null)
    }), watermarkSrc && /*#__PURE__*/ React.createElement(Watermark, {
        src: watermarkSrc
    }), /*#__PURE__*/ React.createElement(CertificateHeader, null, /*#__PURE__*/ React.createElement(LetterheadSection, {
        logoSrc: logoSrc,
        certificateTitle: getTranslation('pdf.vaccineCertificate.title', 'Immunisation Certificate'),
        letterheadConfig: certificateData
    }), /*#__PURE__*/ React.createElement(PatientDetailsSection, {
        patient: patient,
        getLocalisation: getLocalisation,
        getSetting: getSetting,
        certificateId: certificateId,
        extraFields: extraPatientFields
    })), /*#__PURE__*/ React.createElement(Box, {
        style: {
            ...styles.box,
            marginLeft: '18px',
            marginRight: '18px',
            marginBottom: '0px'
        }
    }, /*#__PURE__*/ React.createElement(H3, {
        style: {
            marginBottom: 5,
            marginTop: 5
        }
    }, getTranslation('pdf.vaccineCertificate.immunisationHistory', 'Immunisation history')), /*#__PURE__*/ React.createElement(Table, {
        data: data,
        columns: columns(getTranslation),
        getLocalisation: getLocalisation,
        getSetting: getSetting,
        columnStyle: {
            padding: '10px 5px'
        }
    })), /*#__PURE__*/ React.createElement(SigningSection, {
        signingSrc: signingSrc,
        getTranslation: getTranslation,
        style: {
            marginTop: 30
        },
        wrap: false
    }), /*#__PURE__*/ React.createElement(FixedFooter, null, /*#__PURE__*/ React.createElement(VaccineCertificateFooter, null))));
};
export const VaccineCertificate = withLanguageContext(VaccineCertificateComponent);

//# sourceMappingURL=VaccineCertificate.js.map