"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "STEPS", {
    enumerable: true,
    get: function() {
        return STEPS;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _sync = require("csv-parse/sync");
const _sequelize = require("sequelize");
const _step = require("../step.js");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const STEPS = [
    {
        at: _step.END,
        async check ({ serverType }) {
            // Only run on central server
            return serverType === 'central';
        },
        async run ({ toVersion, models, log }) {
            try {
                const url = `${_config.default.metaServer.host}/versions/${toVersion}/artifacts`;
                log.info('Downloading translation artifacts', {
                    version: toVersion,
                    url
                });
                const artifactsResponse = await fetch(url, {
                    headers: {
                        Accept: 'application/json'
                    }
                });
                if (!artifactsResponse.ok) {
                    throw new Error(`Failed to fetch artifacts list: ${artifactsResponse.status} ${artifactsResponse.statusText}`);
                }
                const artifacts = await artifactsResponse.json();
                const translationsArtifact = artifacts.find((artifact)=>artifact.artifact_type === 'translations');
                if (!translationsArtifact) {
                    throw new Error(`No translations artifact found for version ${toVersion}`);
                }
                log.info('Downloading translations artifact', {
                    url: translationsArtifact.download_url
                });
                const translationsResponse = await fetch(translationsArtifact.download_url);
                if (!translationsResponse.ok) {
                    throw new Error(`Failed to download translations artifact: ${translationsResponse.status} ${translationsResponse.statusText}`);
                }
                const translationRows = (0, _sync.parse)(await translationsResponse.text(), {
                    columns: true,
                    skip_empty_lines: true
                });
                if (translationRows.length === 0) {
                    throw new Error('No valid translation rows found in CSV');
                }
                log.info('Importing new default translations', {
                    count: translationRows.length
                });
                if (translationRows.length > 0) {
                    await models.TranslatedString.sequelize.query(`
                INSERT INTO translated_strings (string_id, text, language)
                VALUES ${translationRows.map(()=>"(?, ?, 'default')").join(', ')}
                ON CONFLICT (string_id, language) DO UPDATE SET text = EXCLUDED.text
              `, {
                        replacements: translationRows.flatMap((row)=>{
                            const text = row.default ?? row.en ?? row.fallback;
                            return text ? [
                                row.stringId,
                                text
                            ] : [];
                        }),
                        type: _sequelize.QueryTypes.INSERT
                    });
                    log.info('Successfully imported default translations');
                }
            } catch (error) {
                // Failing to import translations is not world-ending... for now
                // We may want to make this more strict in the future
                log.error('Failed to import translations, you will need to manually import them', {
                    error
                });
            }
        }
    }
];

//# sourceMappingURL=1752105049000-downloadTranslationArtifacts.js.map